# اصلاح HTML Escape برای Model Name

## 🔥 مشکل واقعی

در لاگ Liara AI:
```
"model":"openai&#x2F;gpt-5-mini"
```

کاراکتر `/` به `&#x2F;` (HTML entity) تبدیل شده است!

## 🔍 علت

`sanitizeBody` middleware در `backend/middleware.js` از `validator.escape()` استفاده می‌کند که همه string ها را HTML-escape می‌کند:

```javascript
const sanitizeString = (str) => {
  return validator.escape(str.trim()); // این / را به &#x2F; تبدیل می‌کند!
};
```

وقتی کاربر در AdminPanel مدل را تنظیم می‌کند و ذخیره می‌شود:
1. `POST /api/settings` با `sanitizeBody` middleware
2. `modelName: "openai/gpt-5-mini:online"` → `"openai&#x2F;gpt-5-mini:online"`
3. در database ذخیره می‌شود: `"openai&#x2F;gpt-5-mini:online"`
4. وقتی خوانده می‌شود: `"openai&#x2F;gpt-5-mini:online"`
5. به Liara AI ارسال می‌شود: ❌ نامعتبر!

## ✅ راه‌حل

### 1. اضافه کردن فیلدهای استثنا در `sanitizeObject`

فیلدهای فنی که نباید HTML-escape شوند:
- `modelName` - نام مدل AI
- `apiKey` - کلید API
- `endpoint` - endpoint های API
- `url`, `href`, `src` - URL ها
- و سایر فیلدهای فنی

### 2. Decode کردن modelName قبل از ارسال

اگر modelName قبلاً HTML-escaped شده، آن را decode کنیم.

## 🔧 تغییرات انجام شده

### 1. backend/middleware.js

اضافه کردن لیست `NO_ESCAPE_FIELDS`:
```javascript
const NO_ESCAPE_FIELDS = [
  'modelName', // AI model names like "openai/gpt-5-mini:online"
  'apiKey', // API keys should not be escaped
  'endpoint', // API endpoints
  'baseURL', // Base URLs
  'url', // URLs in general
  'href', // Links
  'src', // Image sources
  'canonicalUrl', // SEO canonical URLs
  'ogUrl', // Open Graph URLs
  'ogImage', // Open Graph image URLs
  'schemaUrl', // Schema.org URLs
  'schemaLogo', // Schema.org logo URLs
  'favicon', // Favicon URLs
  'appleTouchIcon', // Apple touch icon URLs
  'projectUrl', // Project URLs
];
```

تغییر `sanitizeObject`:
```javascript
if (NO_ESCAPE_FIELDS.includes(key)) {
  // Only trim, don't escape
  sanitized[key] = value.trim();
}
```

### 2. backend/routes.js

اضافه کردن decode برای modelName:
```javascript
// Get model name and ensure it's not HTML-escaped
let modelName = settings?.aiConfig?.modelName || 'openai/gpt-5-mini:online';
// Decode HTML entities if present (fix for previously escaped model names)
if (modelName.includes('&#x2F;') || modelName.includes('&#47;')) {
  modelName = modelName.replace(/&#x2F;/g, '/').replace(/&#47;/g, '/');
}
```

## ✅ نتیجه

1. ✅ فیلدهای فنی دیگر HTML-escape نمی‌شوند
2. ✅ modelName های قبلی که escape شده‌اند، decode می‌شوند
3. ✅ modelName به درستی به Liara AI ارسال می‌شود

## 📝 نکته

اگر در database مدل‌های قبلی HTML-escaped شده‌اند، باید:
1. یا به صورت دستی در AdminPanel دوباره تنظیم شوند
2. یا از decode در کد استفاده شود (که انجام شده)

## ✅ وضعیت: برطرف شده

مشکل HTML escape برای modelName برطرف شد.

