# بررسی ذخیره‌سازی SEO در دیتابیس

## ✅ بله، تغییرات SEO در دیتابیس ذخیره می‌شود

### جریان ذخیره‌سازی:

#### 1. Frontend (SEOManager.tsx)
```typescript
const handleSave = async () => {
  const updatedSettings: AppSettings = {
    ...settings,
    seo: localSEO  // ✅ SEO settings اضافه می‌شود
  };
  
  await DataService.updateSettings(updatedSettings);  // ✅ ارسال به backend
};
```

#### 2. DataService (services/dataService.ts)
```typescript
updateSettings: async (settings: AppSettings) => {
  if (useRealApi()) {
    await apiClient('/settings', 'PUT', settings);  // ✅ ارسال کل settings object
  }
}
```

#### 3. Backend Route (backend/routes.js)
```javascript
router.put('/settings', authenticateToken, sanitizeBody, async (req, res) => {
  const settingsData = JSON.stringify(req.body);  // ✅ تبدیل به JSON string
  
  // ✅ ذخیره در دیتابیس
  if (existing.length > 0) {
    await pool.execute('UPDATE settings SET data = ? WHERE id = ?', 
      [settingsData, existing[0].id]);
  } else {
    await pool.execute('INSERT INTO settings (data) VALUES (?)', [settingsData]);
  }
});
```

#### 4. Database (MySQL)
```sql
CREATE TABLE IF NOT EXISTS settings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  data JSON NOT NULL,  -- ✅ تمام settings (شامل seo) در این ستون JSON ذخیره می‌شود
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
```

### ساختار داده در دیتابیس:

```json
{
  "siteName": "ویراوینگ",
  "seo": {
    "title": "ویراوینگ - طراحی سایت",
    "description": "...",
    "keywords": [...],
    "ogTitle": "...",
    "ogDescription": "...",
    "googleAnalyticsId": "...",
    // ... تمام فیلدهای SEO
  },
  "hero": {...},
  "footer": {...},
  // ... سایر تنظیمات
}
```

### بازیابی از دیتابیس:

#### Backend (GET /settings)
```javascript
router.get('/settings', async (req, res) => {
  const [rows] = await pool.execute('SELECT data FROM settings ORDER BY id DESC LIMIT 1');
  
  if (rows.length > 0) {
    const settings = typeof rows[0].data === 'string' 
      ? JSON.parse(rows[0].data)  // ✅ Parse JSON string
      : rows[0].data;  // ✅ اگر از نوع JSON باشد
    res.json(settings);
  }
});
```

## ✅ نتیجه

**بله، تمام تغییرات SEO در دیتابیس MySQL ذخیره می‌شود:**

1. ✅ SEO settings در object `settings.seo` ذخیره می‌شود
2. ✅ کل `settings` object به صورت JSON در ستون `data` جدول `settings` ذخیره می‌شود
3. ✅ هنگام بازیابی، تمام SEO settings بازگردانده می‌شود
4. ✅ تغییرات پایدار هستند و بعد از restart سرور حفظ می‌شوند

## تست

برای اطمینان می‌توانید:
1. تنظیمات SEO را تغییر دهید و ذخیره کنید
2. صفحه را refresh کنید
3. بررسی کنید که تنظیمات حفظ شده‌اند
4. یا مستقیماً در دیتابیس بررسی کنید:
   ```sql
   SELECT data FROM settings ORDER BY id DESC LIMIT 1;
   ```

