import React, { useState, useEffect } from 'react';
import { Sparkles, ArrowRight, Database, Server, Monitor, Send } from 'lucide-react';
import { generateProjectPlan } from '../services/geminiService';
import { AIPlanResponse, LoadingState, AppSettings } from '../types';
import { DataService } from '../services/dataService';

const AIPlanner: React.FC = () => {
  const [idea, setIdea] = useState('');
  const [loading, setLoading] = useState<LoadingState>(LoadingState.IDLE);
  const [plan, setPlan] = useState<AIPlanResponse | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [settings, setSettings] = useState<AppSettings | null>(null);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getSettings();
      setSettings(data);
    };
    update();
    window.addEventListener('settings-updated', update);
    return () => window.removeEventListener('settings-updated', update);
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!idea.trim() || !settings) return;

    setLoading(LoadingState.LOADING);
    setError(null);
    setPlan(null);

    try {
      // Pass the custom instruction from settings
      const result = await generateProjectPlan(idea, settings.aiConfig.systemInstruction);
      setPlan(result);
      setLoading(LoadingState.SUCCESS);
    } catch (err: any) {
      setError(err.message || 'خطایی رخ داد');
      setLoading(LoadingState.ERROR);
    }
  };

  if (!settings || !settings.aiConfig.enabled) {
    return null;
  }

  return (
    <section id="ai-consultant" className="py-24 bg-gradient-to-br from-indigo-100 via-white to-purple-100 dark:from-indigo-900 dark:via-slate-900 dark:to-purple-900 relative overflow-hidden transition-colors duration-300">
      
      {/* Decorative BG */}
      <div className="absolute inset-0 opacity-10 dark:opacity-20 bg-[url('https://www.transparenttextures.com/patterns/cubes.png')] mix-blend-multiply dark:mix-blend-screen"></div>

      <div className="container mx-auto px-4 relative z-10">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <span className="inline-block py-1 px-3 rounded-full bg-teal-100 dark:bg-teal-500/10 text-teal-700 dark:text-teal-300 text-sm font-bold border border-teal-200 dark:border-teal-500/20 mb-4">
              ✨ قدرت گرفته از Liara AI
            </span>
            <h2 className="text-3xl md:text-5xl font-bold text-slate-900 dark:text-white mb-6">
              مشاور هوشمند پروژه شما
            </h2>
            <p className="text-slate-600 dark:text-slate-300 text-lg leading-relaxed">
              ایده خود را بنویسید (مثلاً: یک سایت فروشگاهی برای فروش لوازم یدکی خودرو). هوش مصنوعی ویراوینگ بهترین استک فنی و نقشه راه را برای شما طراحی می‌کند.
            </p>
          </div>

          {/* Input Form */}
          <div className="bg-white/80 dark:bg-slate-800/60 backdrop-blur-xl border border-white dark:border-slate-700 rounded-2xl p-2 md:p-4 shadow-2xl mb-12">
            <form onSubmit={handleSubmit} className="relative flex flex-col md:flex-row gap-4">
              <input
                type="text"
                value={idea}
                onChange={(e) => setIdea(e.target.value)}
                placeholder="ایده پروژه خود را اینجا بنویسید..."
                className="w-full bg-transparent text-slate-900 dark:text-white text-lg px-6 py-4 outline-none placeholder-slate-400 dark:placeholder-slate-500"
                disabled={loading === LoadingState.LOADING}
              />
              <button
                type="submit"
                disabled={loading === LoadingState.LOADING || !idea.trim()}
                className="md:w-auto w-full px-8 py-4 bg-indigo-600 hover:bg-indigo-500 text-white font-bold rounded-xl transition-all disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2 whitespace-nowrap shadow-lg shadow-indigo-600/20"
              >
                {loading === LoadingState.LOADING ? (
                   <div className="w-6 h-6 border-2 border-white/30 border-t-white rounded-full animate-spin"></div>
                ) : (
                  <>
                    <span>تحلیل ایده</span>
                    <Sparkles size={20} />
                  </>
                )}
              </button>
            </form>
          </div>

          {/* Error Message */}
          {error && (
            <div className="bg-red-50 dark:bg-red-500/10 border border-red-200 dark:border-red-500/50 text-red-600 dark:text-red-200 p-4 rounded-xl mb-8 text-center">
              {error}
            </div>
          )}

          {/* Results Display */}
          {plan && (
            <div className="space-y-8 animate-fade-in-up">
              
              {/* Summary Card */}
              <div className="bg-white/90 dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 rounded-2xl p-8 shadow-xl">
                <h3 className="text-xl font-bold text-indigo-600 dark:text-indigo-300 mb-4 flex items-center gap-2">
                  <Sparkles className="w-5 h-5" />
                  تحلیل کلی
                </h3>
                <p className="text-slate-700 dark:text-slate-300 leading-relaxed text-lg">
                  {plan.summary}
                </p>
              </div>

              {/* Tech Stack Grid */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="bg-white dark:bg-slate-800/50 border border-slate-200 dark:border-slate-700 p-6 rounded-xl hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors shadow-sm">
                  <div className="w-12 h-12 bg-blue-100 dark:bg-blue-500/20 text-blue-600 dark:text-blue-400 rounded-lg flex items-center justify-center mb-4">
                    <Monitor size={24} />
                  </div>
                  <h4 className="text-lg font-bold text-slate-900 dark:text-white mb-2">Frontend</h4>
                  <p className="text-slate-600 dark:text-slate-400 font-mono text-sm">{plan.techStack.frontend}</p>
                </div>

                <div className="bg-white dark:bg-slate-800/50 border border-slate-200 dark:border-slate-700 p-6 rounded-xl hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors shadow-sm">
                  <div className="w-12 h-12 bg-green-100 dark:bg-green-500/20 text-green-600 dark:text-green-400 rounded-lg flex items-center justify-center mb-4">
                    <Server size={24} />
                  </div>
                  <h4 className="text-lg font-bold text-slate-900 dark:text-white mb-2">Backend</h4>
                  <p className="text-slate-600 dark:text-slate-400 font-mono text-sm">{plan.techStack.backend}</p>
                </div>

                <div className="bg-white dark:bg-slate-800/50 border border-slate-200 dark:border-slate-700 p-6 rounded-xl hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors shadow-sm">
                  <div className="w-12 h-12 bg-yellow-100 dark:bg-yellow-500/20 text-yellow-600 dark:text-yellow-400 rounded-lg flex items-center justify-center mb-4">
                    <Database size={24} />
                  </div>
                  <h4 className="text-lg font-bold text-slate-900 dark:text-white mb-2">Database</h4>
                  <p className="text-slate-600 dark:text-slate-400 font-mono text-sm">{plan.techStack.database}</p>
                </div>
              </div>

              {/* Roadmap Timeline */}
              <div className="bg-white/90 dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 rounded-2xl p-8 shadow-xl">
                <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-8 border-b border-slate-200 dark:border-slate-700 pb-4">
                  نقشه راه پیشنهادی
                </h3>
                <div className="space-y-8">
                  {plan.roadmap.map((phase, index) => (
                    <div key={index} className="relative pl-8 md:pl-0 border-r-2 border-indigo-200 dark:border-indigo-500/30 last:border-0 md:border-0">
                      <div className="md:flex items-start gap-6 group">
                        <div className="hidden md:flex flex-col items-center">
                          <div className="w-8 h-8 rounded-full bg-indigo-600 text-white flex items-center justify-center font-bold text-sm shadow-lg shadow-indigo-600/50 z-10">
                            {index + 1}
                          </div>
                          {index !== plan.roadmap.length - 1 && (
                            <div className="w-0.5 h-full bg-indigo-200 dark:bg-indigo-500/30 my-2"></div>
                          )}
                        </div>
                        
                        <div className="flex-1">
                          <div className="flex items-center gap-3 mb-2">
                             <span className="md:hidden w-6 h-6 rounded-full bg-indigo-600 text-white flex items-center justify-center text-xs font-bold shrink-0">
                              {index + 1}
                             </span>
                             <h4 className="text-lg font-bold text-slate-900 dark:text-white">{phase.phase}</h4>
                             <span className="bg-slate-100 dark:bg-slate-700 text-xs px-2 py-1 rounded text-slate-600 dark:text-slate-300 border border-slate-200 dark:border-slate-600">
                               {phase.duration}
                             </span>
                          </div>
                          <p className="text-slate-600 dark:text-slate-400 text-sm leading-relaxed bg-slate-50 dark:bg-slate-900/50 p-4 rounded-xl border border-slate-200 dark:border-slate-700/50">
                            {phase.description}
                          </p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>

              <div className="flex justify-center pt-8">
                 <button className="bg-slate-900 dark:bg-white text-white dark:text-indigo-900 font-bold px-8 py-4 rounded-xl hover:bg-slate-800 dark:hover:bg-slate-200 transition-colors shadow-xl flex items-center gap-3">
                   <Send className="w-5 h-5" />
                   شروع همکاری برای این پروژه
                 </button>
              </div>

            </div>
          )}
        </div>
      </div>
    </section>
  );
};

export default AIPlanner;