import React, { useState, useEffect } from 'react';
import { 
  LayoutDashboard, Briefcase, Mail, Settings, Plus, Trash2, Edit2, Upload, 
  Check, X, Search, Home, FileText, Globe, Users, Star, LogOut, Layers, 
  Loader2, Share2, ChevronLeft, ChevronRight, Image as ImageIcon, BarChart3,
  TrendingUp, MessageSquare, Menu, Shield, Lock, Save, PenTool, BookOpen, Server, RefreshCw, Search as SearchIcon, Eye
} from 'lucide-react';
import { DataService, fireToast } from '../services/dataService';
import { 
  Project, ContactMessage, AppSettings, BlogPost, TeamMember, 
  Review, Service, ProcessStep, MenuItem, MediaItem
} from '../types';
import AdminChart from './AdminChart';
import RichTextEditor from './RichTextEditor';
import ConfirmModal from './ConfirmModal';
import SEOManager from './SEOManager';

// --- SHARED COMPONENTS ---

const Pagination: React.FC<{ currentPage: number; totalPages: number; onPageChange: (page: number) => void }> = ({ currentPage, totalPages, onPageChange }) => {
  if (totalPages <= 1) return null;
  return (
    <div className="flex items-center justify-center gap-2 mt-6">
      <button onClick={() => onPageChange(currentPage - 1)} disabled={currentPage === 1} className="p-2 rounded-lg border border-slate-200 dark:border-slate-700 disabled:opacity-50 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors"><ChevronRight size={16} /></button>
      <span className="text-sm text-slate-600 dark:text-slate-400">صفحه {currentPage} از {totalPages}</span>
      <button onClick={() => onPageChange(currentPage + 1)} disabled={currentPage === totalPages} className="p-2 rounded-lg border border-slate-200 dark:border-slate-700 disabled:opacity-50 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors"><ChevronLeft size={16} /></button>
    </div>
  );
};

const ImageUploader: React.FC<{ currentImage?: string; onUpload: (url: string) => void }> = ({ currentImage, onUpload }) => {
  const [uploading, setUploading] = useState(false);
  
  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setUploading(true);
      try {
        const url = await DataService.uploadFile(file);
        onUpload(url);
      } catch (err) {
        fireToast('خطا در آپلود تصویر', 'error');
      } finally {
        setUploading(false);
      }
    }
  };

  return (
    <div className="flex flex-col gap-3">
       <label className="block text-sm font-medium text-slate-700 dark:text-slate-300">تصویر شاخص</label>
       <div className="flex items-center gap-4">
          <div className="w-24 h-24 bg-slate-100 dark:bg-slate-800 rounded-xl border border-dashed border-slate-300 dark:border-slate-700 flex items-center justify-center overflow-hidden relative group">
             {uploading ? (
               <Loader2 className="animate-spin text-indigo-600" />
             ) : currentImage ? (
               <img src={currentImage} alt="Preview" className="w-full h-full object-cover" />
             ) : (
               <ImageIcon className="text-slate-400" />
             )}
          </div>
          <div>
             <input type="file" id="img-upload" className="hidden" accept="image/*" onChange={handleFileChange} />
             <label htmlFor="img-upload" className="cursor-pointer bg-indigo-50 dark:bg-slate-800 text-indigo-600 dark:text-indigo-400 px-4 py-2 rounded-lg text-sm font-bold hover:bg-indigo-100 dark:hover:bg-slate-700 transition-colors flex items-center gap-2">
               <Upload size={16} />
               {currentImage ? 'تغییر تصویر' : 'آپلود تصویر'}
             </label>
             <p className="text-xs text-slate-400 mt-2">فرمت‌های مجاز: JPG, PNG, WEBP</p>
          </div>
       </div>
    </div>
  );
};

// --- SUB COMPONENTS FOR TABS ---

const MenuManager: React.FC<{ settings: AppSettings | null }> = ({ settings }) => {
    const [headerLinks, setHeaderLinks] = useState<MenuItem[]>([]);
    const [footerLinks, setFooterLinks] = useState<MenuItem[]>([]);
    
    useEffect(() => {
        if (settings) {
            setHeaderLinks(settings.navigation?.header || []);
            setFooterLinks(settings.navigation?.footer?.quickAccess || []);
        }
    }, [settings]);

    const updateItem = (list: MenuItem[], setList: any, id: string, field: keyof MenuItem, value: any) => {
        const newList = list.map(item => item.id === id ? { ...item, [field]: value } : item);
        setList(newList);
    };

    const deleteItem = (list: MenuItem[], setList: any, id: string) => {
        setList(list.filter(item => item.id !== id));
    };

    const addItem = (list: MenuItem[], setList: any) => {
        setList([...list, { id: Date.now().toString(), name: 'لینک جدید', href: '#' }]);
    };

    const saveMenus = async () => {
        if (!settings) return;
        const newSettings = {
            ...settings,
            navigation: {
                ...settings.navigation,
                header: headerLinks,
                footer: {
                    ...settings.navigation.footer,
                    quickAccess: footerLinks
                }
            }
        };
        await DataService.updateSettings(newSettings);
        fireToast('منوها با موفقیت بروزرسانی شدند', 'success');
    };

    return (
        <div className="space-y-8 animate-fade-in">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold dark:text-white">مدیریت منوها</h2>
                <button onClick={saveMenus} className="btn-primary flex items-center gap-2">
                    <Save size={18} /> ذخیره تغییرات
                </button>
            </div>

            <div className="grid md:grid-cols-2 gap-8">
                {/* Header Menu */}
                <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
                    <div className="flex justify-between items-center mb-4">
                        <h3 className="font-bold text-lg dark:text-white">منوی هدر (بالا)</h3>
                        <button onClick={() => addItem(headerLinks, setHeaderLinks)} className="text-indigo-600 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 p-2 rounded"><Plus size={18} /></button>
                    </div>
                    <div className="space-y-3">
                        {headerLinks.map(item => (
                            <div key={item.id} className="flex gap-2 items-center bg-slate-50 dark:bg-slate-900/50 p-2 rounded-lg">
                                <input type="text" className="input-field text-sm" value={item.name} onChange={e => updateItem(headerLinks, setHeaderLinks, item.id, 'name', e.target.value)} placeholder="عنوان" />
                                <input type="text" className="input-field text-sm dir-ltr" value={item.href} onChange={e => updateItem(headerLinks, setHeaderLinks, item.id, 'href', e.target.value)} placeholder="لینک (#home)" />
                                <input type="text" className="input-field text-sm dir-ltr w-20" value={item.anchor || ''} onChange={e => updateItem(headerLinks, setHeaderLinks, item.id, 'anchor', e.target.value)} placeholder="Anchor" />
                                <button onClick={() => deleteItem(headerLinks, setHeaderLinks, item.id)} className="text-red-500 p-2 hover:bg-red-50 rounded"><Trash2 size={16} /></button>
                            </div>
                        ))}
                    </div>
                </div>

                {/* Footer Quick Access */}
                <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
                    <div className="flex justify-between items-center mb-4">
                        <h3 className="font-bold text-lg dark:text-white">دسترسی سریع (فوتر)</h3>
                        <button onClick={() => addItem(footerLinks, setFooterLinks)} className="text-indigo-600 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 p-2 rounded"><Plus size={18} /></button>
                    </div>
                    <div className="space-y-3">
                         {footerLinks.map(item => (
                            <div key={item.id} className="flex gap-2 items-center bg-slate-50 dark:bg-slate-900/50 p-2 rounded-lg">
                                <input type="text" className="input-field text-sm" value={item.name} onChange={e => updateItem(footerLinks, setFooterLinks, item.id, 'name', e.target.value)} placeholder="عنوان" />
                                <input type="text" className="input-field text-sm dir-ltr" value={item.href} onChange={e => updateItem(footerLinks, setFooterLinks, item.id, 'href', e.target.value)} placeholder="لینک" />
                                <button onClick={() => deleteItem(footerLinks, setFooterLinks, item.id)} className="text-red-500 p-2 hover:bg-red-50 rounded"><Trash2 size={16} /></button>
                            </div>
                        ))}
                    </div>
                </div>
            </div>
            <p className="text-sm text-slate-500">نکته: برای اسکرول به بخش‌های صفحه اصلی، در فیلد Anchor شناسه بخش (مثلا services) را وارد کنید.</p>
        </div>
    );
};

const SettingsManager: React.FC<{ settings: AppSettings | null, onUpdate: () => void }> = ({ settings, onUpdate }) => {
    const [localSettings, setLocalSettings] = useState<AppSettings | null>(null);
    const [useApi, setUseApi] = useState(DataService.getApiMode());

    useEffect(() => {
        setLocalSettings(settings);
    }, [settings]);

    if (!localSettings) return null;

    const handleSave = async () => {
        if (!localSettings) return;
        
        // Ensure all required objects exist and preserve existing data
        const settingsToSave: AppSettings = {
            ...localSettings,
            // Ensure hero object exists with all fields preserved
            hero: {
                badge: localSettings.hero?.badge || '',
                title: localSettings.hero?.title || '',
                highlightedWord: localSettings.hero?.highlightedWord || '',
                description: localSettings.hero?.description || '',
                ctaText: localSettings.hero?.ctaText || '',
                ...(localSettings.hero || {})
            },
            // Ensure seo object exists with all fields preserved
            seo: {
                title: localSettings.seo?.title || '',
                description: localSettings.seo?.description || '',
                keywords: Array.isArray(localSettings.seo?.keywords) ? localSettings.seo.keywords : [],
                ...(localSettings.seo || {})
            },
            // Ensure footer object exists with all fields preserved
            footer: {
                contact: {
                    address: localSettings.footer?.contact?.address || '',
                    phone: localSettings.footer?.contact?.phone || '',
                    email: localSettings.footer?.contact?.email || '',
                    ...(localSettings.footer?.contact || {})
                },
                socials: {
                    ...(localSettings.footer?.socials || {})
                },
                ...(localSettings.footer || {})
            },
            // Ensure aiConfig object exists
            aiConfig: {
                enabled: localSettings.aiConfig?.enabled ?? true,
                provider: localSettings.aiConfig?.provider || 'liara',
                ...(localSettings.aiConfig || {})
            },
            // Preserve navigation if exists (don't overwrite)
            navigation: localSettings.navigation || {
                header: [],
                footer: {
                    quickAccess: [],
                    services: []
                },
                mobile: []
            },
            // Preserve legal if exists
            legal: localSettings.legal || {
                terms: '',
                privacy: ''
            }
        };
        
        await DataService.updateSettings(settingsToSave);
        fireToast('تنظیمات با موفقیت ذخیره شد', 'success');
        onUpdate();
    };
    
    const handleApiToggle = () => {
        const newValue = !useApi;
        setUseApi(newValue);
        DataService.setApiMode(newValue);
    };

    return (
        <div className="space-y-8 animate-fade-in">
            <h2 className="text-2xl font-bold dark:text-white">تنظیمات عمومی سایت</h2>
            
            {/* API Config */}
            <div className={`p-6 rounded-2xl border transition-colors ${useApi ? 'bg-indigo-50 dark:bg-indigo-900/20 border-indigo-200 dark:border-indigo-800' : 'bg-slate-50 dark:bg-slate-800 border-slate-200 dark:border-slate-700'}`}>
                <div className="flex justify-between items-center">
                    <div className="flex items-center gap-3">
                        <div className={`p-3 rounded-xl ${useApi ? 'bg-indigo-500 text-white' : 'bg-slate-300 dark:bg-slate-700 text-slate-600 dark:text-slate-400'}`}>
                            <Server size={24} />
                        </div>
                        <div>
                            <h3 className="font-bold text-lg dark:text-white">منبع داده (Backend API)</h3>
                            <p className="text-sm text-slate-500 dark:text-slate-400">
                                {useApi ? 'متصل به سرور Node.js (http://localhost:3001)' : 'استفاده از داده‌های آزمایشی (LocalStorage)'}
                            </p>
                        </div>
                    </div>
                    <button 
                        onClick={handleApiToggle}
                        className={`px-4 py-2 rounded-lg font-bold text-sm transition-colors ${useApi ? 'bg-red-100 text-red-600 hover:bg-red-200' : 'bg-indigo-600 text-white hover:bg-indigo-700'}`}
                    >
                        {useApi ? 'غیرفعال‌سازی API' : 'فعال‌سازی API'}
                    </button>
                </div>
                {useApi && <p className="mt-4 text-xs text-indigo-600 dark:text-indigo-300 bg-white/50 dark:bg-black/20 p-2 rounded">نکته: مطمئن شوید سرور Node.js در پورت ۳۰۰۱ در حال اجراست، در غیر این صورت با خطای Network Error مواجه می‌شوید.</p>}
            </div>

            {/* Branding */}
            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
               <h3 className="font-bold text-lg mb-4 flex items-center gap-2 dark:text-white"><Globe size={20} className="text-indigo-500" /> هویت بصری</h3>
               <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                     <input type="text" className="input-field" placeholder="نام سایت" value={localSettings.siteName} onChange={e => setLocalSettings({...localSettings, siteName: e.target.value})} />
                     <input type="text" className="input-field" placeholder="عنوان سئو (Title)" value={localSettings.seo?.title || ''} onChange={e => setLocalSettings({...localSettings, seo: {...(localSettings.seo || {}), title: e.target.value}})} />
                     <textarea className="input-field" placeholder="توضیحات سئو (Meta Description)" rows={3} value={localSettings.seo?.description || ''} onChange={e => setLocalSettings({...localSettings, seo: {...(localSettings.seo || {}), description: e.target.value}})}></textarea>
                  </div>
                  <div className="space-y-4">
                     <div className="text-sm dark:text-slate-400">لوگو سایت</div>
                     <div className="space-y-2">
                     <ImageUploader currentImage={localSettings.logoUrl} onUpload={(url) => setLocalSettings({...localSettings, logoUrl: url})} />
                       <button
                         type="button"
                         onClick={() => openMediaPicker((url) => setLocalSettings({...localSettings, logoUrl: url}))}
                         className="text-sm text-indigo-600 hover:underline"
                       >
                         انتخاب از رسانه‌ها
                       </button>
                     </div>
                  </div>
               </div>
            </div>

            {/* NEW: Hero Section Config */}
            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
               <h3 className="font-bold text-lg mb-4 flex items-center gap-2 dark:text-white"><PenTool size={20} className="text-purple-500" /> بخش اصلی (Hero)</h3>
               <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                     <div>
                       <label className="text-xs text-slate-500 mb-1 block">متن بج (Badge)</label>
                       <input type="text" className="input-field" placeholder="مثال: آینده وب اینجاست" value={localSettings.hero?.badge || ''} onChange={e => setLocalSettings({...localSettings, hero: {...(localSettings.hero || {}), badge: e.target.value}})} />
                     </div>
                     <div>
                       <label className="text-xs text-slate-500 mb-1 block">عنوان اصلی</label>
                       <input type="text" className="input-field" placeholder="مثال: ما رویاهای دیجیتال شما را" value={localSettings.hero?.title || ''} onChange={e => setLocalSettings({...localSettings, hero: {...(localSettings.hero || {}), title: e.target.value}})} />
                     </div>
                     <div>
                       <label className="text-xs text-slate-500 mb-1 block">کلمه برجسته (هایلایت)</label>
                       <input type="text" className="input-field" placeholder="مثال: خلق می‌کنیم" value={localSettings.hero?.highlightedWord || ''} onChange={e => setLocalSettings({...localSettings, hero: {...(localSettings.hero || {}), highlightedWord: e.target.value}})} />
                     </div>
                  </div>
                  <div className="space-y-4">
                     <div>
                       <label className="text-xs text-slate-500 mb-1 block">توضیحات زیر عنوان</label>
                       <textarea className="input-field" rows={4} placeholder="توضیحات کوتاه..." value={localSettings.hero?.description || ''} onChange={e => setLocalSettings({...localSettings, hero: {...(localSettings.hero || {}), description: e.target.value}})}></textarea>
                     </div>
                     <div>
                       <label className="text-xs text-slate-500 mb-1 block">متن دکمه اصلی</label>
                       <input type="text" className="input-field" placeholder="مثال: مشاوره رایگان" value={localSettings.hero?.ctaText || ''} onChange={e => setLocalSettings({...localSettings, hero: {...(localSettings.hero || {}), ctaText: e.target.value}})} />
                     </div>
                  </div>
               </div>
            </div>

            {/* Contact Info */}
            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
               <h3 className="font-bold text-lg mb-4 flex items-center gap-2 dark:text-white"><Mail size={20} className="text-green-500" /> اطلاعات تماس</h3>
               <div className="grid md:grid-cols-2 gap-4">
                  <input type="text" className="input-field" placeholder="شماره تماس" value={localSettings.footer?.contact?.phone || ''} onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), contact: {...(localSettings.footer?.contact || {}), phone: e.target.value}}})} />
                  <input type="text" className="input-field" placeholder="ایمیل" value={localSettings.footer?.contact?.email || ''} onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), contact: {...(localSettings.footer?.contact || {}), email: e.target.value}}})} />
                  <input type="text" className="input-field md:col-span-2" placeholder="آدرس" value={localSettings.footer?.contact?.address || ''} onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), contact: {...(localSettings.footer?.contact || {}), address: e.target.value}}})} />
               </div>
            </div>

            {/* Social Media */}
            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
               <h3 className="font-bold text-lg mb-4 flex items-center gap-2 dark:text-white">
                 <Share2 size={20} className="text-blue-500" /> شبکه‌های اجتماعی
               </h3>
               <div className="grid md:grid-cols-2 gap-4">
                  <input 
                    type="text" 
                    className="input-field dir-ltr" 
                    placeholder="Instagram URL" 
                    value={localSettings.footer?.socials?.instagram || ''} 
                    onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), socials: {...(localSettings.footer?.socials || {}), instagram: e.target.value}}})} 
                  />
                  <input 
                    type="text" 
                    className="input-field dir-ltr" 
                    placeholder="Twitter (X) URL" 
                    value={localSettings.footer?.socials?.twitter || ''} 
                    onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), socials: {...(localSettings.footer?.socials || {}), twitter: e.target.value}}})} 
                  />
                  <input 
                    type="text" 
                    className="input-field dir-ltr" 
                    placeholder="LinkedIn URL" 
                    value={localSettings.footer?.socials?.linkedin || ''} 
                    onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), socials: {...(localSettings.footer?.socials || {}), linkedin: e.target.value}}})} 
                  />
                  <input 
                    type="text" 
                    className="input-field dir-ltr" 
                    placeholder="Facebook URL" 
                    value={localSettings.footer?.socials?.facebook || ''} 
                    onChange={e => setLocalSettings({...localSettings, footer: {...(localSettings.footer || {}), socials: {...(localSettings.footer?.socials || {}), facebook: e.target.value}}})} 
                  />
               </div>
            </div>

            {/* AI Config */}
            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700">
               <h3 className="font-bold text-lg mb-4 flex items-center gap-2 dark:text-white"><Star size={20} className="text-yellow-500" /> تنظیمات هوش مصنوعی</h3>
               <div className="space-y-4">
                 <div className="flex items-center gap-2">
                    <input type="checkbox" checked={localSettings.aiConfig?.enabled ?? true} onChange={e => setLocalSettings({...localSettings, aiConfig: {...(localSettings.aiConfig || {}), enabled: e.target.checked}})} className="w-5 h-5 accent-indigo-600" />
                    <span className="dark:text-white">فعال‌سازی دستیار هوشمند</span>
                 </div>
                 <div>
                   <label className="block text-xs text-slate-500 mb-1 dark:text-slate-400">API Key (Liara AI)</label>
                   <input type="password" className="input-field" placeholder="کلید API Liara AI را وارد کنید" value={localSettings.aiConfig?.apiKey || ''} onChange={e => setLocalSettings({...localSettings, aiConfig: {...(localSettings.aiConfig || {}), apiKey: e.target.value}})} />
                   <p className="text-xs text-slate-500 mt-1 dark:text-slate-400">کلید API را از پنل Liara AI دریافت کنید</p>
                 </div>
                 <div>
                   <label className="block text-xs text-slate-500 mb-1 dark:text-slate-400">نام مدل (Model Name)</label>
                   <input type="text" className="input-field dir-ltr" placeholder="openai/gpt-5-mini:online" value={localSettings.aiConfig?.modelName || ''} onChange={e => setLocalSettings({...localSettings, aiConfig: {...(localSettings.aiConfig || {}), modelName: e.target.value}})} />
                   <p className="text-xs text-slate-500 mt-1 dark:text-slate-400">مثال: openai/gpt-5-mini:online یا google/gemini-2.5-flash:online</p>
                 </div>
                 <div>
                   <label className="block text-xs text-slate-500 mb-1 dark:text-slate-400">دستورالعمل سیستم (System Prompt)</label>
                   <textarea className="input-field" placeholder="دستورالعمل سیستم (System Prompt)" rows={3} value={localSettings.aiConfig?.systemInstruction || ''} onChange={e => setLocalSettings({...localSettings, aiConfig: {...(localSettings.aiConfig || {}), systemInstruction: e.target.value}})}></textarea>
                 </div>
               </div>
            </div>

            <button onClick={handleSave} className="w-full py-4 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-xl shadow-lg transition-all sticky bottom-4">
               ذخیره تمام تنظیمات
            </button>
        </div>
    );
};

const LegalManager: React.FC<{ settings: AppSettings | null, onUpdate: () => void }> = ({ settings, onUpdate }) => {
    const [localSettings, setLocalSettings] = useState<AppSettings | null>(null);
    const [activeSection, setActiveSection] = useState<'terms' | 'privacy'>('terms');

    useEffect(() => {
        setLocalSettings(settings);
    }, [settings]);

    if (!localSettings) return null;

    const handleSave = async () => {
        if (!localSettings) return;
        await DataService.updateSettings(localSettings);
        fireToast('محتوای قانونی با موفقیت ذخیره شد', 'success');
        onUpdate();
    };

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold dark:text-white">صفحات قانونی</h2>
                <button onClick={handleSave} className="btn-primary flex items-center gap-2">
                    <Save size={18} /> ذخیره تغییرات
                </button>
            </div>

            {/* Tabs */}
            <div className="flex gap-2 bg-white dark:bg-slate-800 p-1 rounded-xl border border-slate-200 dark:border-slate-700 w-fit">
                <button 
                    onClick={() => setActiveSection('terms')}
                    className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${activeSection === 'terms' ? 'bg-indigo-100 dark:bg-indigo-900/30 text-indigo-600 dark:text-indigo-400' : 'text-slate-600 dark:text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-700'}`}
                >
                    قوانین و مقررات
                </button>
                <button 
                    onClick={() => setActiveSection('privacy')}
                    className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${activeSection === 'privacy' ? 'bg-indigo-100 dark:bg-indigo-900/30 text-indigo-600 dark:text-indigo-400' : 'text-slate-600 dark:text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-700'}`}
                >
                    حریم خصوصی
                </button>
            </div>

            <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 shadow-sm">
                <div className="mb-4">
                    <h3 className="font-bold text-lg dark:text-white mb-2">
                        {activeSection === 'terms' ? 'متن قوانین و مقررات' : 'متن حریم خصوصی'}
                    </h3>
                    <p className="text-sm text-slate-500 dark:text-slate-400">
                        {activeSection === 'terms' 
                            ? 'این متن در صفحه قوانین و مقررات (/terms) نمایش داده می‌شود.' 
                            : 'این متن در صفحه حریم خصوصی (/privacy) نمایش داده می‌شود.'}
                    </p>
                </div>
                
                <RichTextEditor 
                    value={activeSection === 'terms' ? (localSettings.legal?.terms || '') : (localSettings.legal?.privacy || '')}
                    onChange={(val) => {
                        const newLegal = { ...localSettings.legal, [activeSection]: val };
                        setLocalSettings({ ...localSettings, legal: newLegal });
                    }}
                    placeholder="متن کامل را اینجا وارد کنید..."
                />
            </div>
        </div>
    );
};

const SecurityManager: React.FC = () => {
    const [formData, setFormData] = useState({ username: '', password: '', confirm: '' });

    const handleUpdate = async (e: React.FormEvent) => {
        e.preventDefault();
        if (formData.password && formData.password !== formData.confirm) {
            fireToast('تکرار رمز عبور مطابقت ندارد', 'error');
            return;
        }
        
        const updateData: any = {};
        if (formData.username) updateData.username = formData.username;
        if (formData.password) updateData.password = formData.password;

        if (Object.keys(updateData).length === 0) return;

        const success = await DataService.updateAdmin(updateData);
        if (success) {
            fireToast('اطلاعات با موفقیت بروز شد. لطفا مجددا وارد شوید.', 'success');
            setTimeout(() => DataService.logout(), 2000);
        } else {
            fireToast('خطا در بروزرسانی اطلاعات', 'error');
        }
    };

    return (
        <div className="max-w-2xl mx-auto animate-fade-in">
            <h2 className="text-2xl font-bold dark:text-white mb-6">امنیت و پروفایل مدیر</h2>
            <div className="bg-white dark:bg-slate-800 p-8 rounded-2xl border border-slate-200 dark:border-slate-700 shadow-sm">
                <div className="flex flex-col items-center mb-8">
                    <div className="w-20 h-20 bg-slate-100 dark:bg-slate-700 rounded-full flex items-center justify-center text-slate-500 dark:text-slate-400 mb-4">
                        <Shield size={40} />
                    </div>
                    <p className="text-slate-600 dark:text-slate-400 text-center text-sm">
                        تغییر نام کاربری و رمز عبور پنل مدیریت.<br/>
                        توجه: پس از تغییر، باید مجددا وارد شوید.
                    </p>
                </div>

                <form onSubmit={handleUpdate} className="space-y-6">
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-2">نام کاربری جدید</label>
                        <input type="text" className="input-field dir-ltr" placeholder="نام کاربری فعلی را نگه دارید..." value={formData.username} onChange={e => setFormData({...formData, username: e.target.value})} />
                    </div>
                    
                    <div className="grid md:grid-cols-2 gap-4">
                        <div>
                            <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-2">رمز عبور جدید</label>
                            <div className="relative">
                                <Lock size={16} className="absolute top-3 left-3 text-slate-400" />
                                <input type="password" className="input-field dir-ltr pl-10" placeholder="••••••" value={formData.password} onChange={e => setFormData({...formData, password: e.target.value})} />
                            </div>
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-2">تکرار رمز عبور</label>
                            <div className="relative">
                                <Lock size={16} className="absolute top-3 left-3 text-slate-400" />
                                <input type="password" className="input-field dir-ltr pl-10" placeholder="••••••" value={formData.confirm} onChange={e => setFormData({...formData, confirm: e.target.value})} />
                            </div>
                        </div>
                    </div>

                    <button type="submit" className="w-full btn-primary py-3 flex items-center justify-center gap-2">
                        <Save size={18} />
                        بروزرسانی اطلاعات امنیتی
                    </button>
                </form>
            </div>
        </div>
    );
};

// --- MAIN ADMIN COMPONENT ---

const AdminPanel: React.FC = () => {
  const [activeTab, setActiveTab] = useState<'dashboard' | 'projects' | 'messages' | 'settings' | 'blog' | 'team' | 'reviews' | 'services' | 'process' | 'menus' | 'security' | 'legal' | 'media' | 'seo'>('dashboard');
  
  // Data States
  const [projects, setProjects] = useState<Project[]>([]);
  const [messages, setMessages] = useState<ContactMessage[]>([]);
  const [blogPosts, setBlogPosts] = useState<BlogPost[]>([]);
  const [team, setTeam] = useState<TeamMember[]>([]);
  const [reviews, setReviews] = useState<Review[]>([]);
  const [servicesData, setServicesData] = useState<Service[]>([]);
  const [processSteps, setProcessSteps] = useState<ProcessStep[]>([]);
  const [settings, setSettings] = useState<AppSettings | null>(null);
  const [mediaItems, setMediaItems] = useState<MediaItem[]>([]);
  
  // Counts & Filter
  const [counts, setCounts] = useState({ projects: 0, blog: 0, messages: 0 }); // Dashboard badge counts
  const [filteredTotal, setFilteredTotal] = useState(0); // For pagination
  const [searchTerm, setSearchTerm] = useState('');

  // UI States
  const [loading, setLoading] = useState(false);
  const [mediaLoading, setMediaLoading] = useState(false);
  const [mediaModalOpen, setMediaModalOpen] = useState(false);
  const [mediaSelectHandler, setMediaSelectHandler] = useState<((url: string) => void) | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 6;

  // Forms & Editing States
  const [editingProject, setEditingProject] = useState<Partial<Project> | null>(null);
  const [editingBlog, setEditingBlog] = useState<Partial<BlogPost> | null>(null);
  const [editingTeam, setEditingTeam] = useState<Partial<TeamMember> | null>(null);
  const [editingReview, setEditingReview] = useState<Partial<Review> | null>(null);
  const [editingService, setEditingService] = useState<Partial<Service> | null>(null);
  const [editingProcess, setEditingProcess] = useState<Partial<ProcessStep> | null>(null);
  
  // Confirm Modal State
  const [confirmModal, setConfirmModal] = useState<{
    isOpen: boolean;
    title: string;
    message: string;
    onConfirm: () => void;
    variant?: 'danger' | 'warning' | 'info';
  }>({
    isOpen: false,
    title: '',
    message: '',
    onConfirm: () => {},
    variant: 'danger'
  });

  useEffect(() => {
    refreshData();
    window.addEventListener('data-updated', refreshData);
    return () => window.removeEventListener('data-updated', refreshData);
  }, [activeTab, currentPage, searchTerm]);

  const refreshData = async () => {
    setLoading(true);
    
    // Always fetch settings and dashboard stats
    const currentSettings = await DataService.getSettings();
    setSettings(currentSettings);
    
    setCounts(await DataService.getDashboardStats());
    
    if (activeTab === 'dashboard') {
        const recentMsgs = await DataService.getMessages(1, 5);
        setMessages(recentMsgs.data);
        
        // Fetch all projects for stats calculation
        const allProjects = await DataService.getProjects(1, 100);
        setProjects(allProjects.data);
    }
    else if (activeTab === 'projects') {
        const res = await DataService.getProjects(currentPage, itemsPerPage, searchTerm);
        setProjects(res.data);
        setFilteredTotal(res.total);
    }
    else if (activeTab === 'blog') {
        const res = await DataService.getBlogPosts(currentPage, itemsPerPage, searchTerm);
        setBlogPosts(res.data);
        setFilteredTotal(res.total);
    }
    else if (activeTab === 'messages') {
        const res = await DataService.getMessages(currentPage, itemsPerPage);
        setMessages(res.data);
        setFilteredTotal(res.total);
    }
    else if (activeTab === 'team') setTeam(await DataService.getTeam());
    else if (activeTab === 'reviews') setReviews(await DataService.getReviews());
    else if (activeTab === 'services') setServicesData(await DataService.getServices());
    else if (activeTab === 'process') setProcessSteps(await DataService.getProcessSteps());
    else if (activeTab === 'media') {
        setMediaLoading(true);
        try {
          const media = await DataService.getMedia();
          setMediaItems(media);
        } catch (e) {
          fireToast('خطا در بارگذاری رسانه‌ها', 'error');
        }
        setMediaLoading(false);
    }
    
    setLoading(false);
  };

  const openMediaPicker = async (onSelect: (url: string) => void) => {
    setMediaSelectHandler(() => onSelect);
    // Load media if not loaded
    if (mediaItems.length === 0) {
      setMediaLoading(true);
      try {
        const media = await DataService.getMedia();
        setMediaItems(media);
      } catch (e) {
        fireToast('خطا در بارگذاری رسانه‌ها', 'error');
      }
      setMediaLoading(false);
    }
    setMediaModalOpen(true);
  };

  const handleLogout = () => {
    DataService.logout();
  };

  const handleDelete = (type: string, id: string, itemName?: string) => {
    const typeNames: Record<string, string> = {
      project: 'پروژه',
      blog: 'مقاله',
      team: 'عضو تیم',
      review: 'نظر',
      service: 'خدمت',
      process: 'مرحله',
      message: 'پیام'
    };
    
    setConfirmModal({
      isOpen: true,
      title: 'تأیید حذف',
      message: `آیا از حذف ${typeNames[type] || 'این آیتم'}${itemName ? ` "${itemName}"` : ''} مطمئن هستید؟ این عمل غیرقابل بازگشت است.`,
      variant: 'danger',
      onConfirm: async () => {
        setConfirmModal({ ...confirmModal, isOpen: false });
    setLoading(true);
    try {
        if(type === 'project') await DataService.deleteProject(id);
        if(type === 'blog') await DataService.deleteBlogPost(id);
        if(type === 'team') await DataService.deleteTeamMember(id);
        if(type === 'review') await DataService.deleteReview(id);
        if(type === 'service') await DataService.deleteService(id);
        if(type === 'process') await DataService.deleteProcessStep(id);
        if(type === 'message') await DataService.deleteMessage(id);
        fireToast('حذف با موفقیت انجام شد', 'success');
        await refreshData();
        } catch(e: any) {
          logger.error('Delete error in admin panel', { error: e.message, type, id });
        fireToast('خطا در حذف آیتم', 'error');
        } finally {
    setLoading(false);
        }
      }
    });
  };

  const calculateProjectStats = () => {
    const stats: Record<string, number> = {};
    projects.forEach(p => {
        const cat = p.category || 'سایر';
        stats[cat] = (stats[cat] || 0) + 1;
    });
    return Object.entries(stats).map(([label, value]) => ({ label, value }));
  };

  const renderDashboard = () => {
      const projectStats = calculateProjectStats();

      return (
          <div className="space-y-8 animate-fade-in">
              <h2 className="text-2xl font-bold dark:text-white mb-6">داشبورد مدیریت</h2>
              
              {/* Stats Cards */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                  <div className="bg-gradient-to-br from-indigo-600 to-purple-700 rounded-2xl p-6 text-white shadow-lg shadow-indigo-600/30">
                      <div className="flex justify-between items-start mb-4">
                          <div className="p-3 bg-white/20 rounded-xl"><BarChart3 size={24} /></div>
                          <span className="text-xs bg-white/20 px-2 py-1 rounded backdrop-blur-sm">+۱۲٪</span>
                      </div>
                      <h3 className="text-3xl font-bold mb-1">۱,۲۳۴</h3>
                      <p className="text-indigo-200 text-sm">بازدید کل هفته</p>
                  </div>

                  <div className="bg-white dark:bg-slate-800 rounded-2xl p-6 border border-slate-200 dark:border-slate-700 shadow-sm">
                      <div className="flex justify-between items-start mb-4">
                          <div className="p-3 bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 rounded-xl"><Briefcase size={24} /></div>
                      </div>
                      <h3 className="text-3xl font-bold dark:text-white mb-1">{counts.projects}</h3>
                      <p className="text-slate-500 text-sm">پروژه‌های فعال</p>
                  </div>

                  <div className="bg-white dark:bg-slate-800 rounded-2xl p-6 border border-slate-200 dark:border-slate-700 shadow-sm">
                      <div className="flex justify-between items-start mb-4">
                          <div className="p-3 bg-green-100 dark:bg-green-900/30 text-green-600 dark:text-green-400 rounded-xl"><FileText size={24} /></div>
                      </div>
                      <h3 className="text-3xl font-bold dark:text-white mb-1">{counts.blog}</h3>
                      <p className="text-slate-500 text-sm">مقالات منتشر شده</p>
                  </div>

                  <div className="bg-white dark:bg-slate-800 rounded-2xl p-6 border border-slate-200 dark:border-slate-700 shadow-sm">
                      <div className="flex justify-between items-start mb-4">
                          <div className="p-3 bg-yellow-100 dark:bg-yellow-900/30 text-yellow-600 dark:text-yellow-400 rounded-xl"><MessageSquare size={24} /></div>
                      </div>
                      <h3 className="text-3xl font-bold dark:text-white mb-1">{counts.messages}</h3>
                      <p className="text-slate-500 text-sm">پیام‌های دریافتی</p>
                  </div>
              </div>

              <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                  {/* Chart */}
                  <div className="lg:col-span-1">
                      <AdminChart title="دسته‌بندی پروژه‌ها" data={projectStats} />
                  </div>

                  {/* Recent Messages */}
                  <div className="lg:col-span-2 bg-white dark:bg-slate-800 rounded-2xl border border-slate-200 dark:border-slate-700 p-6 shadow-sm flex flex-col">
                      <div className="flex justify-between items-center mb-6">
                          <h3 className="font-bold text-lg dark:text-white">آخرین پیام‌ها</h3>
                          <button onClick={() => setActiveTab('messages')} className="text-indigo-600 text-sm hover:underline">مشاهده همه</button>
                      </div>
                      <div className="space-y-4 flex-1">
                          {messages.length === 0 ? (
                              <div className="h-full flex items-center justify-center text-slate-500 text-sm">پیامی وجود ندارد</div>
                          ) : messages.slice(0, 5).map(msg => (
                              <div key={msg.id} className="flex items-center justify-between p-3 bg-slate-50 dark:bg-slate-700/30 rounded-xl hover:bg-slate-100 dark:hover:bg-slate-700/50 transition-colors">
                                  <div className="flex items-center gap-3">
                                      <div className={`w-2 h-2 rounded-full ${msg.isRead ? 'bg-slate-300' : 'bg-red-500'}`}></div>
                                      <div>
                                          <h4 className="font-bold text-sm dark:text-white">{msg.name}</h4>
                                          <p className="text-xs text-slate-500 truncate max-w-[200px]">{msg.description}</p>
                                      </div>
                                  </div>
                                  <span className="text-xs text-slate-400">{msg.date}</span>
                              </div>
                          ))}
                      </div>
                  </div>
              </div>

              {/* Quick Actions */}
               <div className="bg-white dark:bg-slate-800 rounded-2xl border border-slate-200 dark:border-slate-700 p-6 shadow-sm">
                      <h3 className="font-bold text-lg dark:text-white mb-6">دسترسی سریع</h3>
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                          <button onClick={() => { setActiveTab('projects'); setEditingProject({}); }} className="flex items-center justify-center gap-2 p-4 bg-slate-50 dark:bg-slate-700/50 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 hover:text-indigo-600 border border-slate-200 dark:border-slate-700 rounded-xl transition-all text-slate-700 dark:text-slate-300 font-medium">
                              <Plus size={18} />
                              <span>پروژه جدید</span>
                          </button>
                          <button onClick={() => { setActiveTab('blog'); setEditingBlog({}); }} className="flex items-center justify-center gap-2 p-4 bg-slate-50 dark:bg-slate-700/50 hover:bg-green-50 dark:hover:bg-green-900/20 hover:text-green-600 border border-slate-200 dark:border-slate-700 rounded-xl transition-all text-slate-700 dark:text-slate-300 font-medium">
                              <Plus size={18} />
                              <span>مقاله جدید</span>
                          </button>
                          <button onClick={() => setActiveTab('settings')} className="flex items-center justify-center gap-2 p-4 bg-slate-50 dark:bg-slate-700/50 hover:bg-slate-100 dark:hover:bg-slate-700 border border-slate-200 dark:border-slate-700 rounded-xl transition-all text-slate-700 dark:text-slate-300 font-medium">
                              <Settings size={18} />
                              <span>تنظیمات</span>
                          </button>
                      </div>
                  </div>
          </div>
      )
  }

  const renderSearch = () => {
      if (!['projects', 'blog'].includes(activeTab)) return null;
      return (
          <div className="relative mb-6">
              <Search className="absolute right-3 top-3 text-slate-400" size={20} />
              <input 
                 type="text" 
                 placeholder="جستجو..." 
                 value={searchTerm}
                 onChange={(e) => { setSearchTerm(e.target.value); setCurrentPage(1); }}
                 className="w-full bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 rounded-xl px-10 py-2.5 outline-none focus:border-indigo-500 transition-colors"
              />
          </div>
      )
  };

  const renderProjects = () => {
    const totalPages = Math.ceil(filteredTotal / itemsPerPage);

    if (editingProject) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
          <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingProject.id ? 'ویرایش پروژه' : 'افزودن پروژه جدید'}</h3>
            <button onClick={() => setEditingProject(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveProject(editingProject as Project);
            setEditingProject(null);
            fireToast('پروژه با موفقیت ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <div className="grid md:grid-cols-2 gap-4">
              <input type="text" placeholder="عنوان پروژه" required className="input-field" value={editingProject.title || ''} onChange={e => setEditingProject({...editingProject, title: e.target.value})} />
              <input type="text" placeholder="دسته‌بندی (Web, App...)" required className="input-field" value={editingProject.category || ''} onChange={e => setEditingProject({...editingProject, category: e.target.value})} />
            </div>
            <div className="grid md:grid-cols-2 gap-4">
               <input type="text" placeholder="نام مشتری" className="input-field" value={editingProject.client || ''} onChange={e => setEditingProject({...editingProject, client: e.target.value})} />
               <input type="text" placeholder="سال اجرا" className="input-field" value={editingProject.year || ''} onChange={e => setEditingProject({...editingProject, year: e.target.value})} />
            </div>
            <input
              type="text"
              placeholder="لینک پروژه (https://example.com)"
              className="input-field dir-ltr"
              value={editingProject.projectUrl || ''}
              onChange={e => setEditingProject({ ...editingProject, projectUrl: e.target.value })}
            />
            <textarea placeholder="توضیحات کامل پروژه" rows={4} className="input-field" value={editingProject.description || ''} onChange={e => setEditingProject({...editingProject, description: e.target.value})}></textarea>
            <input type="text" placeholder="تکنولوژی‌ها (با کاما جدا کنید)" className="input-field" value={Array.isArray(editingProject.tags) ? editingProject.tags.join(', ') : editingProject.tags || ''} onChange={e => setEditingProject({...editingProject, tags: e.target.value.split(',').map(t=>t.trim())})} />
            
            <div className="space-y-2">
            <ImageUploader currentImage={editingProject.image} onUpload={(url) => setEditingProject({...editingProject, image: url})} />
              <button
                type="button"
                onClick={() => openMediaPicker((url) => setEditingProject({...editingProject, image: url}))}
                className="text-sm text-indigo-600 hover:underline"
              >
                انتخاب از رسانه‌ها
              </button>
            </div>
            
            <div className="flex justify-end gap-2 mt-6">
               <button type="button" onClick={() => setEditingProject(null)} className="px-4 py-2 text-slate-500">انصراف</button>
               <button type="submit" className="px-6 py-2 bg-indigo-600 text-white rounded-xl hover:bg-indigo-700">ذخیره تغییرات</button>
            </div>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت پروژه‌ها</h2>
          <button onClick={() => setEditingProject({})} className="btn-primary flex items-center gap-2">
            <Plus size={18} /> افزودن پروژه
          </button>
        </div>
        
        {renderSearch()}

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {projects.map(p => (
            <div key={p.id} className="bg-white dark:bg-slate-800 rounded-xl overflow-hidden shadow-sm border border-slate-200 dark:border-slate-700 group">
              <div className="h-40 overflow-hidden relative">
                <img src={p.image} alt={p.title} className="w-full h-full object-cover" />
                <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-2">
                  <button onClick={() => setEditingProject(p)} className="p-2 bg-white rounded-full text-indigo-600"><Edit2 size={16} /></button>
                  <button onClick={() => handleDelete('project', p.id, p.title)} className="p-2 bg-white rounded-full text-red-500"><Trash2 size={16} /></button>
                </div>
              </div>
              <div className="p-4">
                <h3 className="font-bold dark:text-white">{p.title}</h3>
                <p className="text-xs text-slate-500 mt-1">{p.category}</p>
              </div>
            </div>
          ))}
        </div>
        {projects.length === 0 && <div className="text-center py-10 text-slate-500">موردی یافت نشد.</div>}
        <Pagination currentPage={currentPage} totalPages={totalPages} onPageChange={setCurrentPage} />
      </div>
    );
  };

  const renderBlog = () => {
    const totalPages = Math.ceil(filteredTotal / itemsPerPage);

    if (editingBlog) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
           <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingBlog.id ? 'ویرایش مقاله' : 'افزودن مقاله'}</h3>
            <button onClick={() => setEditingBlog(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveBlogPost(editingBlog as BlogPost);
            setEditingBlog(null);
            fireToast('مقاله ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <input type="text" placeholder="عنوان مقاله" required className="input-field" value={editingBlog.title || ''} onChange={e => setEditingBlog({...editingBlog, title: e.target.value})} />
            <input type="text" placeholder="خلاصه کوتاه" className="input-field" value={editingBlog.excerpt || ''} onChange={e => setEditingBlog({...editingBlog, excerpt: e.target.value})} />
            <div className="grid grid-cols-2 gap-4">
              <input type="text" placeholder="نویسنده" className="input-field" value={editingBlog.author || ''} onChange={e => setEditingBlog({...editingBlog, author: e.target.value})} />
              <input type="text" placeholder="زمان مطالعه (مثلا ۵ دقیقه)" className="input-field" value={editingBlog.readTime || ''} onChange={e => setEditingBlog({...editingBlog, readTime: e.target.value})} />
            </div>
            <div>
              <label className="block text-sm text-slate-500 mb-1">محتوای مقاله</label>
              <RichTextEditor 
                value={editingBlog.content || ''} 
                onChange={(val) => setEditingBlog({...editingBlog, content: val})} 
                placeholder="متن مقاله را اینجا بنویسید (پشتیبانی از فرمت‌های HTML)" 
              />
            </div>
            <div className="space-y-2">
            <ImageUploader currentImage={editingBlog.image} onUpload={(url) => setEditingBlog({...editingBlog, image: url})} />
              <button
                type="button"
                onClick={() => openMediaPicker((url) => setEditingBlog({...editingBlog, image: url}))}
                className="text-sm text-indigo-600 hover:underline"
              >
                انتخاب از رسانه‌ها
              </button>
            </div>
            <button type="submit" className="w-full btn-primary py-3">ذخیره مقاله</button>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت وبلاگ</h2>
          <button onClick={() => setEditingBlog({})} className="btn-primary flex items-center gap-2"><Plus size={18} /> افزودن مقاله</button>
        </div>
        
        {renderSearch()}

        <div className="space-y-3">
          {blogPosts.map(post => (
            <div key={post.id} className="flex items-center gap-4 bg-white dark:bg-slate-800 p-4 rounded-xl border border-slate-200 dark:border-slate-700">
               <img src={post.image} alt={post.title} className="w-16 h-16 rounded-lg object-cover" />
               <div className="flex-1">
                 <h3 className="font-bold dark:text-white">{post.title}</h3>
                 <p className="text-xs text-slate-500">{post.date} • {post.author}</p>
               </div>
               <div className="flex gap-2">
                 <button onClick={() => setEditingBlog(post)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded-lg text-indigo-500"><Edit2 size={18} /></button>
                 <button onClick={() => handleDelete('blog', post.id, post.title)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded-lg text-red-500"><Trash2 size={18} /></button>
               </div>
            </div>
          ))}
        </div>
         {blogPosts.length === 0 && <div className="text-center py-10 text-slate-500">موردی یافت نشد.</div>}
        <Pagination currentPage={currentPage} totalPages={totalPages} onPageChange={setCurrentPage} />
      </div>
    );
  };

  const renderMessages = () => {
    const totalPages = Math.ceil(filteredTotal / itemsPerPage);

    return (
      <div className="space-y-6">
        <h2 className="text-2xl font-bold dark:text-white">پیام‌های دریافتی</h2>
        {messages.length === 0 ? (
          <div className="text-center text-slate-500 py-10">پیام جدیدی وجود ندارد</div>
        ) : (
          <div className="space-y-4">
             {messages.map(msg => (
               <div key={msg.id} className={`p-5 rounded-2xl border transition-all ${msg.isRead ? 'bg-white dark:bg-slate-800 border-slate-200 dark:border-slate-700' : 'bg-indigo-50 dark:bg-indigo-900/20 border-indigo-200 dark:border-indigo-800'}`}>
                  <div className="flex justify-between items-start mb-3">
                     <div>
                       <h3 className="font-bold text-lg dark:text-white">{msg.name}</h3>
                       <p className="text-sm text-slate-500 font-mono">{msg.phone}</p>
                     </div>
                     <span className="text-xs text-slate-400">{msg.date}</span>
                  </div>
                  <div className="flex gap-2 mb-3">
                    {msg.services.map(s => <span key={s} className="px-2 py-0.5 bg-white dark:bg-slate-700 rounded text-xs border border-slate-200 dark:border-slate-600">{s}</span>)}
                  </div>
                  <p className="text-slate-700 dark:text-slate-300 text-sm mb-4 bg-white/50 dark:bg-slate-900/50 p-3 rounded-lg">{msg.description}</p>
                  <div className="flex gap-3 justify-end">
                     {!msg.isRead && (
                       <button onClick={async () => { await DataService.markAsRead(msg.id); refreshData(); fireToast('پیام خوانده شد', 'info'); }} className="text-xs flex items-center gap-1 text-indigo-600 hover:underline">
                         <Check size={14} /> خواندم
                       </button>
                     )}
                     <button onClick={() => handleDelete('message', msg.id)} className="text-xs flex items-center gap-1 text-red-500 hover:underline">
                       <Trash2 size={14} /> حذف
                     </button>
                  </div>
               </div>
             ))}
             <Pagination currentPage={currentPage} totalPages={totalPages} onPageChange={setCurrentPage} />
          </div>
        )}
      </div>
    );
  };

  const renderTeam = () => {
    if (editingTeam) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
           <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingTeam.id ? 'ویرایش عضو تیم' : 'افزودن عضو تیم'}</h3>
            <button onClick={() => setEditingTeam(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveTeamMember(editingTeam as TeamMember);
            setEditingTeam(null);
            fireToast('عضو تیم ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <div className="grid md:grid-cols-2 gap-4">
              <input type="text" placeholder="نام و نام خانوادگی" required className="input-field" value={editingTeam.name || ''} onChange={e => setEditingTeam({...editingTeam, name: e.target.value})} />
              <input type="text" placeholder="سمت (Role)" required className="input-field" value={editingTeam.role || ''} onChange={e => setEditingTeam({...editingTeam, role: e.target.value})} />
            </div>
            <textarea placeholder="بیوگرافی کوتاه" className="input-field" rows={3} value={editingTeam.bio || ''} onChange={e => setEditingTeam({...editingTeam, bio: e.target.value})}></textarea>
            
            <div className="grid md:grid-cols-3 gap-4">
               <input type="text" placeholder="LinkedIn URL" className="input-field" value={editingTeam.socials?.linkedin || ''} onChange={e => setEditingTeam({...editingTeam, socials: {...editingTeam.socials, linkedin: e.target.value}})} />
               <input type="text" placeholder="Github URL" className="input-field" value={editingTeam.socials?.github || ''} onChange={e => setEditingTeam({...editingTeam, socials: {...editingTeam.socials, github: e.target.value}})} />
               <input type="text" placeholder="Twitter URL" className="input-field" value={editingTeam.socials?.twitter || ''} onChange={e => setEditingTeam({...editingTeam, socials: {...editingTeam.socials, twitter: e.target.value}})} />
            </div>

            <div className="space-y-2">
            <ImageUploader currentImage={editingTeam.image} onUpload={(url) => setEditingTeam({...editingTeam, image: url})} />
              <button
                type="button"
                onClick={() => openMediaPicker((url) => setEditingTeam({...editingTeam, image: url}))}
                className="text-sm text-indigo-600 hover:underline"
              >
                انتخاب از رسانه‌ها
              </button>
            </div>
            
            <button type="submit" className="w-full btn-primary py-3">ذخیره</button>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت تیم</h2>
          <button onClick={() => setEditingTeam({})} className="btn-primary flex items-center gap-2"><Plus size={18} /> افزودن عضو</button>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {team.map(member => (
            <div key={member.id} className="bg-white dark:bg-slate-800 rounded-xl p-4 border border-slate-200 dark:border-slate-700 flex items-center gap-4">
               <img src={member.image} alt={member.name} className="w-16 h-16 rounded-full object-cover" />
               <div className="flex-1">
                 <h3 className="font-bold dark:text-white">{member.name}</h3>
                 <p className="text-xs text-slate-500">{member.role}</p>
               </div>
               <div className="flex flex-col gap-2">
                 <button onClick={() => setEditingTeam(member)} className="text-indigo-500 hover:bg-slate-100 dark:hover:bg-slate-700 p-2 rounded"><Edit2 size={16} /></button>
                     <button onClick={() => handleDelete('team', member.id, member.name)} className="text-red-500 hover:bg-slate-100 dark:hover:bg-slate-700 p-2 rounded"><Trash2 size={16} /></button>
               </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderReviews = () => {
    if (editingReview) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
           <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingReview.id ? 'ویرایش نظر' : 'افزودن نظر مشتری'}</h3>
            <button onClick={() => setEditingReview(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveReview(editingReview as Review);
            setEditingReview(null);
            fireToast('نظر ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <div className="grid md:grid-cols-2 gap-4">
              <input type="text" placeholder="نام مشتری" required className="input-field" value={editingReview.name || ''} onChange={e => setEditingReview({...editingReview, name: e.target.value})} />
              <input type="text" placeholder="سمت (Role)" className="input-field" value={editingReview.role || ''} onChange={e => setEditingReview({...editingReview, role: e.target.value})} />
            </div>
            <textarea placeholder="متن نظر" required className="input-field" rows={4} value={editingReview.text || ''} onChange={e => setEditingReview({...editingReview, text: e.target.value})}></textarea>
            
            <div className="space-y-2">
            <ImageUploader currentImage={editingReview.avatar} onUpload={(url) => setEditingReview({...editingReview, avatar: url})} />
              <button
                type="button"
                onClick={() => openMediaPicker((url) => setEditingReview({...editingReview, avatar: url}))}
                className="text-sm text-indigo-600 hover:underline"
              >
                انتخاب از رسانه‌ها
              </button>
            </div>
            
            <button type="submit" className="w-full btn-primary py-3">ذخیره</button>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت نظرات</h2>
          <button onClick={() => setEditingReview({})} className="btn-primary flex items-center gap-2"><Plus size={18} /> افزودن نظر</button>
        </div>
        <div className="space-y-4">
          {reviews.map(review => (
            <div key={review.id} className="bg-white dark:bg-slate-800 rounded-xl p-4 border border-slate-200 dark:border-slate-700 flex items-start gap-4">
               <img src={review.avatar} alt={review.name} className="w-12 h-12 rounded-full object-cover" />
               <div className="flex-1">
                 <div className="flex justify-between mb-2">
                   <div>
                     <h3 className="font-bold dark:text-white">{review.name}</h3>
                     <p className="text-xs text-slate-500">{review.role}</p>
                   </div>
                   <div className="flex gap-2">
                     <button onClick={() => setEditingReview(review)} className="text-indigo-500"><Edit2 size={16} /></button>
                     <button onClick={() => handleDelete('review', review.id, review.name)} className="text-red-500"><Trash2 size={16} /></button>
                   </div>
                 </div>
                 <p className="text-sm text-slate-600 dark:text-slate-400 bg-slate-50 dark:bg-slate-900/50 p-3 rounded-lg">"{review.text}"</p>
               </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderServices = () => {
    if (editingService) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
           <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingService.id ? 'ویرایش خدمت' : 'افزودن خدمت'}</h3>
            <button onClick={() => setEditingService(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveService(editingService as Service);
            setEditingService(null);
            fireToast('خدمت ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <input type="text" placeholder="عنوان خدمت" required className="input-field" value={editingService.title || ''} onChange={e => setEditingService({...editingService, title: e.target.value})} />
            <textarea placeholder="توضیحات کوتاه" className="input-field" rows={3} value={editingService.description || ''} onChange={e => setEditingService({...editingService, description: e.target.value})}></textarea>
            <input type="text" placeholder="نام آیکون (مثال: Globe, Smartphone, Palette...)" className="input-field dir-ltr" value={editingService.iconName || ''} onChange={e => setEditingService({...editingService, iconName: e.target.value})} />
            <p className="text-xs text-slate-400">آیکون‌های مجاز: Globe, Smartphone, Palette, Search, Server, ShieldCheck, Zap, Code, Database, Layout</p>
            
            <button type="submit" className="w-full btn-primary py-3">ذخیره</button>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت خدمات</h2>
          <button onClick={() => setEditingService({})} className="btn-primary flex items-center gap-2"><Plus size={18} /> افزودن خدمت</button>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {servicesData.map(service => (
            <div key={service.id} className="bg-white dark:bg-slate-800 rounded-xl p-6 border border-slate-200 dark:border-slate-700 relative group">
               <h3 className="font-bold dark:text-white mb-2">{service.title}</h3>
               <p className="text-sm text-slate-500 mb-4">{service.description}</p>
               <span className="text-xs bg-slate-100 dark:bg-slate-700 px-2 py-1 rounded text-slate-500">Icon: {service.iconName}</span>
               
               <div className="absolute top-4 left-4 flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                 <button onClick={() => setEditingService(service)} className="text-indigo-500"><Edit2 size={16} /></button>
                 <button onClick={() => handleDelete('service', service.id, service.title)} className="text-red-500"><Trash2 size={16} /></button>
               </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const renderProcess = () => {
    if (editingProcess) {
      return (
        <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl shadow-sm animate-fade-in">
           <div className="flex justify-between items-center mb-6">
            <h3 className="text-xl font-bold dark:text-white">{editingProcess.id ? 'ویرایش مرحله' : 'افزودن مرحله'}</h3>
            <button onClick={() => setEditingProcess(null)} className="text-slate-500 hover:text-red-500"><X /></button>
          </div>
          <form onSubmit={async (e) => {
            e.preventDefault();
            await DataService.saveProcessStep(editingProcess as ProcessStep);
            setEditingProcess(null);
            fireToast('مرحله ذخیره شد', 'success');
            refreshData();
          }} className="space-y-4">
            <input type="text" placeholder="عنوان مرحله" required className="input-field" value={editingProcess.title || ''} onChange={e => setEditingProcess({...editingProcess, title: e.target.value})} />
            <textarea placeholder="توضیحات" className="input-field" rows={3} value={editingProcess.description || ''} onChange={e => setEditingProcess({...editingProcess, description: e.target.value})}></textarea>
            <input type="text" placeholder="نام آیکون" className="input-field dir-ltr" value={editingProcess.iconName || ''} onChange={e => setEditingProcess({...editingProcess, iconName: e.target.value})} />
            
            <button type="submit" className="w-full btn-primary py-3">ذخیره</button>
          </form>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">مدیریت مراحل کار</h2>
          <button onClick={() => setEditingProcess({})} className="btn-primary flex items-center gap-2"><Plus size={18} /> افزودن مرحله</button>
        </div>
        <div className="space-y-4">
          {processSteps.map((step, index) => (
            <div key={step.id} className="bg-white dark:bg-slate-800 rounded-xl p-4 border border-slate-200 dark:border-slate-700 flex items-center gap-4">
               <div className="w-8 h-8 rounded-full bg-indigo-100 dark:bg-indigo-900/50 text-indigo-600 flex items-center justify-center font-bold">
                 {index + 1}
               </div>
               <div className="flex-1">
                 <h3 className="font-bold dark:text-white">{step.title}</h3>
                 <p className="text-sm text-slate-500">{step.description}</p>
               </div>
               <div className="flex gap-2">
                 <button onClick={() => setEditingProcess(step)} className="text-indigo-500 p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded"><Edit2 size={16} /></button>
                 <button onClick={() => handleDelete('process', step.id)} className="text-red-500 p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded"><Trash2 size={16} /></button>
               </div>
            </div>
          ))}
        </div>
      </div>
    );
  };

  const formatSize = (bytes: number) => {
    if (bytes < 1024) return `${bytes} B`;
    if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)} KB`;
    return `${(bytes / (1024 * 1024)).toFixed(1)} MB`;
  };

  const renderMediaPickerModal = () => {
    if (!mediaModalOpen) return null;
    return (
      <div className="fixed inset-0 z-50 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4">
        <div className="bg-white dark:bg-slate-900 rounded-2xl shadow-2xl border border-slate-200 dark:border-slate-700 w-full max-w-5xl max-h-[80vh] overflow-hidden flex flex-col">
          <div className="flex items-center justify-between px-6 py-4 border-b border-slate-200 dark:border-slate-800">
            <h3 className="text-lg font-bold dark:text-white">انتخاب از رسانه‌ها</h3>
            <div className="flex gap-2">
              <button
                onClick={async () => {
                  setMediaLoading(true);
                  try {
                    const media = await DataService.getMedia();
                    setMediaItems(media);
                  } catch (e) {
                    fireToast('خطا در بروزرسانی رسانه‌ها', 'error');
                  }
                  setMediaLoading(false);
                }}
                className="p-2 rounded-lg bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 transition-colors"
              >
                <RefreshCw size={16} className={mediaLoading ? 'animate-spin' : ''} />
              </button>
              <button
                onClick={() => setMediaModalOpen(false)}
                className="p-2 rounded-lg bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 transition-colors"
              >
                <X size={18} />
              </button>
            </div>
          </div>
          <div className="p-4 overflow-auto flex-1">
            {mediaLoading ? (
              <div className="flex items-center gap-2 text-slate-500 dark:text-slate-300">
                <Loader2 className="animate-spin" />
                در حال بارگذاری رسانه‌ها...
              </div>
            ) : mediaItems.length === 0 ? (
              <div className="text-center py-10 text-slate-500">
                رسانه‌ای یافت نشد. ابتدا تصویر آپلود کنید.
              </div>
            ) : (
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                {mediaItems.map((item) => (
                  <div
                    key={item.filename}
                    className="bg-white dark:bg-slate-800 rounded-xl border border-slate-200 dark:border-slate-700 overflow-hidden shadow-sm group"
                  >
                    <div className="aspect-video bg-slate-100 dark:bg-slate-900 overflow-hidden">
                      <img src={item.url} alt={item.filename} className="w-full h-full object-cover" />
                    </div>
                    <div className="p-3 space-y-2 text-sm">
                      <div className="font-semibold text-slate-800 dark:text-white truncate">{item.filename}</div>
                      <div className="text-xs text-slate-500 dark:text-slate-400 flex justify-between">
                        <span>{formatSize(item.size)}</span>
                        <span>{new Date(item.createdAt).toLocaleDateString('fa-IR')}</span>
                      </div>
                      <div className="flex gap-2">
                        <button
                          onClick={() => {
                            if (mediaSelectHandler) mediaSelectHandler(item.url);
                            setMediaModalOpen(false);
                          }}
                          className="flex-1 px-3 py-2 rounded-lg bg-indigo-600 text-white text-xs hover:bg-indigo-700 transition-colors"
                        >
                          انتخاب
                        </button>
                        <button
                          onClick={() => {
                            setConfirmModal({
                              isOpen: true,
                              title: 'تأیید حذف فایل',
                              message: `آیا از حذف فایل "${item.filename}" مطمئن هستید؟ این عمل غیرقابل بازگشت است.`,
                              variant: 'danger',
                              onConfirm: async () => {
                                setConfirmModal({ ...confirmModal, isOpen: false });
                                try {
                                  await DataService.deleteMedia(item.filename);
                                  setMediaItems((prev) => prev.filter((m) => m.filename !== item.filename));
                                  fireToast('فایل حذف شد', 'success');
                                } catch (e: any) {
                                  fireToast('خطا در حذف فایل', 'error');
                                }
                              }
                            });
                          }}
                          className="px-3 py-2 rounded-lg text-red-500 hover:bg-red-50 dark:hover:bg-red-900/10 text-xs transition-colors"
                        >
                          حذف
                        </button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>
      </div>
    );
  };
  const renderMedia = () => {
    return (
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold dark:text-white">رسانه‌ها (آپلود شده‌ها)</h2>
          <button
            onClick={refreshData}
            className="btn-primary flex items-center gap-2"
            disabled={mediaLoading}
          >
            <RefreshCw size={16} className={mediaLoading ? 'animate-spin' : ''} />
            بروزرسانی
          </button>
        </div>

        {mediaLoading ? (
          <div className="flex items-center gap-2 text-slate-500 dark:text-slate-300">
            <Loader2 className="animate-spin" />
            در حال بارگذاری رسانه‌ها...
          </div>
        ) : mediaItems.length === 0 ? (
          <div className="text-center py-10 text-slate-500">
            رسانه‌ای یافت نشد. ابتدا از بخش‌های مختلف تصویر آپلود کنید.
          </div>
        ) : (
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {mediaItems.map((item) => (
              <div
                key={item.filename}
                className="bg-white dark:bg-slate-800 rounded-2xl border border-slate-200 dark:border-slate-700 overflow-hidden shadow-sm group relative"
              >
                <div className="aspect-video w-full overflow-hidden bg-slate-100 dark:bg-slate-900">
                  <img src={item.url} alt={item.filename} className="w-full h-full object-cover" />
                </div>
                <div className="p-4 space-y-2">
                  <div className="text-sm font-bold text-slate-800 dark:text-white truncate">{item.filename}</div>
                  <div className="text-xs text-slate-500 dark:text-slate-400 flex justify-between">
                    <span>{formatSize(item.size)}</span>
                    <span>{new Date(item.createdAt).toLocaleDateString('fa-IR')}</span>
                  </div>
                  <button
                    onClick={() => {
                      setConfirmModal({
                        isOpen: true,
                        title: 'تأیید حذف فایل',
                        message: `آیا از حذف فایل "${item.filename}" مطمئن هستید؟ این عمل غیرقابل بازگشت است.`,
                        variant: 'danger',
                        onConfirm: async () => {
                          setConfirmModal({ ...confirmModal, isOpen: false });
                          try {
                            await DataService.deleteMedia(item.filename);
                            fireToast('فایل حذف شد', 'success');
                            setMediaItems((prev) => prev.filter((m) => m.filename !== item.filename));
                          } catch (e: any) {
                            fireToast('خطا در حذف فایل', 'error');
                          }
                        }
                      });
                    }}
                    className="w-full mt-2 flex items-center justify-center gap-2 text-red-500 hover:bg-red-50 dark:hover:bg-red-900/10 px-3 py-2 rounded-lg transition-colors"
                  >
                    <Trash2 size={16} />
                    حذف فایل
                  </button>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    );
  };

  return (
    <div className="min-h-screen pt-20 bg-slate-100 dark:bg-slate-950 flex transition-colors duration-300 relative">
      <style>{`
        .input-field {
           width: 100%;
           background-color: rgb(248 250 252);
           border: 1px solid rgb(226 232 240);
           border-radius: 0.75rem;
           padding: 0.75rem 1rem;
           outline: none;
           transition: all;
        }
        .dark .input-field {
           background-color: rgb(30 41 59);
           border-color: rgb(51 65 85);
           color: white;
        }
        .input-field:focus {
           border-color: #4f46e5;
           box-shadow: 0 0 0 2px rgba(79, 70, 229, 0.1);
        }
        .dir-ltr {
           direction: ltr;
           text-align: left;
        }
        .btn-primary {
           background-color: #4f46e5;
           color: white;
           padding: 0.5rem 1rem;
           border-radius: 0.75rem;
           font-weight: 500;
           transition: background-color 0.2s;
        }
        .btn-primary:hover {
           background-color: #4338ca;
        }
      `}</style>

      {/* Back to Home (Fixed) */}
      <a 
        href="#home"
        onClick={(e) => { e.preventDefault(); window.location.hash = '#home'; window.location.reload(); }}
        className="fixed top-4 left-4 z-50 bg-white dark:bg-slate-800 text-slate-600 dark:text-slate-300 px-4 py-2 rounded-xl shadow-lg border border-slate-200 dark:border-slate-700 flex items-center gap-2 hover:bg-indigo-600 hover:text-white dark:hover:bg-indigo-600 dark:hover:text-white transition-colors"
      >
        <Home size={18} />
        <span className="hidden md:inline">بازگشت به سایت</span>
      </a>

      {/* Sidebar Navigation */}
      <aside className="w-64 bg-white dark:bg-slate-900 border-l border-slate-200 dark:border-slate-800 fixed top-0 right-0 h-full hidden md:flex flex-col pt-4 z-40 shadow-xl">
         <div className="p-6">
            <h2 className="text-2xl font-black text-slate-800 dark:text-white">پنل مدیریت</h2>
         </div>
         <nav className="mt-4 px-4 space-y-2 pb-8 flex-1 overflow-y-auto">
            {[
                {id: 'dashboard', label: 'داشبورد', icon: LayoutDashboard},
                {id: 'projects', label: 'نمونه کارها', icon: Briefcase, count: counts.projects},
                {id: 'services', label: 'خدمات', icon: Layers},
                {id: 'process', label: 'پروسه کاری', icon: Share2},
                {id: 'blog', label: 'وبلاگ', icon: FileText, count: counts.blog},
                {id: 'team', label: 'متخصصان', icon: Users},
                {id: 'reviews', label: 'نظرات', icon: Star},
                {id: 'messages', label: 'پیام‌ها', icon: Mail, count: counts.messages},
                {id: 'menus', label: 'مدیریت منوها', icon: Menu},
                {id: 'legal', label: 'قوانین و مقررات', icon: BookOpen},
                {id: 'media', label: 'رسانه‌ها', icon: ImageIcon},
                {id: 'seo', label: 'SEO', icon: SearchIcon},
                {id: 'settings', label: 'تنظیمات', icon: Settings},
                {id: 'security', label: 'امنیت', icon: Shield},
            ].map(item => (
                <button 
                key={item.id}
                onClick={() => { setActiveTab(item.id as any); setCurrentPage(1); setSearchTerm(''); }}
                className={`w-full flex items-center gap-3 px-4 py-3 rounded-xl transition-colors ${activeTab === item.id ? 'bg-indigo-600 text-white shadow-lg shadow-indigo-500/30' : 'text-slate-600 dark:text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-800'}`}
                >
                <item.icon size={20} /> 
                {item.label}
                {item.count ? <span className="mr-auto bg-slate-200 dark:bg-slate-700 text-slate-600 dark:text-slate-300 text-xs px-2 py-0.5 rounded-full">{item.count}</span> : null}
                </button>
            ))}
         </nav>
         
         <div className="p-4 border-t border-slate-100 dark:border-slate-800">
            <button onClick={handleLogout} className="w-full flex items-center gap-3 px-4 py-3 rounded-xl text-red-500 hover:bg-red-50 dark:hover:bg-red-900/10 transition-colors">
               <LogOut size={20} /> خروج از حساب
            </button>
         </div>
      </aside>

      {/* Main Content Area */}
      <main className="flex-1 md:mr-64 p-4 md:p-10 overflow-auto w-full relative">
        {loading && (
            <div className="absolute inset-0 bg-white/50 dark:bg-slate-900/50 flex items-center justify-center z-50 backdrop-blur-sm rounded-3xl">
                <Loader2 className="w-12 h-12 animate-spin text-indigo-600" />
            </div>
        )}
        
        {/* Mobile Tabs */}
        <div className="md:hidden flex overflow-x-auto gap-2 mb-6 pb-2 pt-4 hide-scrollbar">
             {[
                {id: 'dashboard', label: 'داشبورد'},
                {id: 'projects', label: 'پروژه‌ها'},
                {id: 'messages', label: 'پیام‌ها'},
                {id: 'blog', label: 'وبلاگ'},
                {id: 'menus', label: 'منوها'},
                {id: 'legal', label: 'قوانین'},
                {id: 'media', label: 'رسانه‌ها'},
                {id: 'seo', label: 'SEO'},
                {id: 'settings', label: 'تنظیمات'},
             ].map(item => (
                <button 
                  key={item.id} 
                  onClick={() => setActiveTab(item.id as any)}
                  className={`whitespace-nowrap px-4 py-2 rounded-lg text-sm font-medium ${activeTab === item.id ? 'bg-indigo-600 text-white' : 'bg-white dark:bg-slate-800 text-slate-600 dark:text-slate-400'}`}
                >
                    {item.label}
                </button>
            ))}
        </div>

        {/* Content Renderers */}
        <div className="animate-fade-in-up pb-24">
            {activeTab === 'dashboard' && renderDashboard()}
            {activeTab === 'projects' && renderProjects()}
            {activeTab === 'blog' && renderBlog()}
            {activeTab === 'messages' && renderMessages()}
            {activeTab === 'settings' && <SettingsManager settings={settings} onUpdate={refreshData} />}
            {activeTab === 'team' && renderTeam()}
            {activeTab === 'reviews' && renderReviews()}
            {activeTab === 'services' && renderServices()}
            {activeTab === 'process' && renderProcess()}
            {activeTab === 'menus' && <MenuManager settings={settings} />}
            {activeTab === 'media' && renderMedia()}
            {activeTab === 'legal' && <LegalManager settings={settings} onUpdate={refreshData} />}
            {activeTab === 'seo' && <SEOManager settings={settings} onUpdate={refreshData} />}
            {activeTab === 'security' && <SecurityManager />}
        </div>
        
        {/* Confirm Modal */}
        <ConfirmModal
          isOpen={confirmModal.isOpen}
          title={confirmModal.title}
          message={confirmModal.message}
          variant={confirmModal.variant}
          onConfirm={confirmModal.onConfirm}
          onCancel={() => setConfirmModal({ ...confirmModal, isOpen: false })}
        />
        
        {renderMediaPickerModal()}
      </main>
    </div>
  );
};

export default AdminPanel;