import React, { useState, useEffect } from 'react';
import { ArrowLeft, Calendar, Clock } from 'lucide-react';
import { DataService } from '../services/dataService';
import { BlogPost } from '../types';

const Blog: React.FC = () => {
  const [posts, setPosts] = useState<BlogPost[]>([]);

  useEffect(() => {
    const fetchData = async () => {
      // Optimized: Fetch only 3 posts for the homepage widget
      const response = await DataService.getBlogPosts(1, 3);
      setPosts(response.data);
    };
    fetchData();
    window.addEventListener('data-updated', fetchData);
    return () => window.removeEventListener('data-updated', fetchData);
  }, []);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  return (
    <section className="py-16 md:py-24 bg-white dark:bg-slate-950 transition-colors duration-300">
      <div className="container mx-auto md:px-4">
        <div className="flex justify-between items-end mb-8 md:mb-12 px-4">
          <div>
            <h2 className="text-2xl md:text-5xl font-bold text-slate-900 dark:text-white mb-2 md:mb-4">آخرین مقالات</h2>
            <div className="h-1 w-16 md:w-24 bg-indigo-600 rounded-full"></div>
          </div>
          <button onClick={(e) => handleNavClick(e as any, '#blog')} className="hidden md:flex text-slate-500 dark:text-slate-400 hover:text-indigo-600 dark:hover:text-white items-center gap-2 transition-colors cursor-pointer">
            مشاهده وبلاگ
            <ArrowLeft size={20} />
          </button>
        </div>

        {/* Mobile: Horizontal Scroll | Desktop: Grid */}
        <div className="flex overflow-x-auto gap-4 px-4 pb-8 snap-x snap-mandatory hide-scrollbar md:grid md:grid-cols-3 md:gap-8 md:pb-0">
          {posts.map((post) => (
            <article 
              key={post.id} 
              className="min-w-[280px] w-[80vw] md:w-auto snap-center flex-shrink-0 group cursor-pointer"
            >
              <div className="relative overflow-hidden rounded-3xl mb-4 aspect-video shadow-md">
                <img 
                  src={post.image} 
                  alt={post.title} 
                  className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                />
                <div className="absolute inset-0 bg-black/20 group-hover:bg-black/10 transition-colors"></div>
                
                {/* Mobile overlay text for better app feel */}
                <div className="absolute bottom-2 left-3 right-3 flex items-center justify-between text-[10px] text-white/90 bg-black/40 backdrop-blur-sm px-2 py-1 rounded-lg md:hidden">
                    <span className="flex items-center gap-1"><Calendar size={10} /> {post.date}</span>
                    <span className="flex items-center gap-1"><Clock size={10} /> {post.readTime}</span>
                </div>
              </div>
              
              <div className="hidden md:flex items-center gap-4 text-xs text-slate-500 mb-3">
                <span className="flex items-center gap-1"><Calendar size={14} /> {post.date}</span>
                <span className="flex items-center gap-1"><Clock size={14} /> {post.readTime}</span>
              </div>
              
              <h3 className="text-lg md:text-xl font-bold text-slate-900 dark:text-white mb-2 group-hover:text-indigo-600 dark:group-hover:text-indigo-400 transition-colors leading-tight line-clamp-2">
                {post.title}
              </h3>
              
              <p className="text-slate-600 dark:text-slate-400 text-sm leading-relaxed mb-4 line-clamp-2">
                {post.excerpt}
              </p>
              
              <a 
                href={`#blog/${post.id}`} 
                onClick={(e) => handleNavClick(e, `#blog/${post.id}`)}
                className="text-indigo-600 dark:text-indigo-500 text-sm font-medium flex items-center gap-1 group-hover:gap-2 transition-all"
              >
                ادامه مطلب
                <ArrowLeft size={16} />
              </a>
            </article>
          ))}
          {/* Spacer */}
          <div className="w-1 flex-shrink-0 md:hidden"></div>
        </div>
        
        <div className="mt-4 text-center md:hidden">
            <button onClick={(e) => handleNavClick(e as any, '#blog')} className="text-slate-500 hover:text-indigo-600 items-center gap-2 inline-flex transition-colors text-sm font-bold p-2 bg-slate-50 dark:bg-slate-900 rounded-xl cursor-pointer">
            مشاهده همه مقالات
            <ArrowLeft size={16} />
          </button>
        </div>
      </div>
    </section>
  );
};

export default Blog;