import React, { useEffect, useState } from 'react';
import { Calendar, Clock, ArrowLeft, Loader2, PlusCircle, Search } from 'lucide-react';
import { DataService } from '../services/dataService';
import { BlogPost } from '../types';
import { BlogSkeleton } from './Skeletons';

const BlogList: React.FC = () => {
  const [posts, setPosts] = useState<BlogPost[]>([]);
  const [page, setPage] = useState(1);
  const [hasMore, setHasMore] = useState(true);
  const [loading, setLoading] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const LIMIT = 6;

  const fetchPosts = async (reset: boolean = false) => {
      const targetPage = reset ? 1 : page;
      if (reset) {
          setLoading(true);
          setPosts([]);
      } else {
          setLoadingMore(true);
      }

      // Simulate network delay for skeleton demo
      if(reset) await new Promise(r => setTimeout(r, 800));

      const response = await DataService.getBlogPosts(targetPage, LIMIT, searchTerm);
      
      if (reset) {
          setPosts(response.data);
      } else {
          setPosts(prev => [...prev, ...response.data]);
      }

      if (response.data.length < LIMIT || (reset ? response.data.length : posts.length + response.data.length) >= response.total) {
          setHasMore(false);
      } else {
          setHasMore(true);
      }

      if (reset) setPage(1);
      setLoading(false);
      setLoadingMore(false);
  };

  useEffect(() => {
    fetchPosts(true);
  }, []);

  // Search debounce
  useEffect(() => {
      const delayDebounceFn = setTimeout(() => {
        fetchPosts(true);
      }, 500);
      return () => clearTimeout(delayDebounceFn);
  }, [searchTerm]);

  const loadMore = async () => {
    if (!hasMore || loadingMore) return;
    const nextPage = page + 1;
    setPage(nextPage);
    
    setLoadingMore(true);
    const response = await DataService.getBlogPosts(nextPage, LIMIT, searchTerm);
    
    setPosts(prev => [...prev, ...response.data]);
    
    if (response.data.length < LIMIT || posts.length + response.data.length >= response.total) {
        setHasMore(false);
    }
    setLoadingMore(false);
  };

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  return (
    <section className="min-h-screen pt-24 md:pt-32 pb-20 bg-slate-50 dark:bg-slate-950 transition-colors duration-300">
      <div className="container mx-auto px-4">
        
        <div className="text-center mb-10">
          <span className="text-indigo-600 dark:text-indigo-400 font-bold tracking-wider text-sm uppercase">Blog</span>
          <h1 className="text-3xl md:text-5xl font-bold text-slate-900 dark:text-white mt-2 mb-4">مقالات و دانستنی‌ها</h1>
          <p className="text-slate-600 dark:text-slate-400 max-w-2xl mx-auto">
            جدیدترین اخبار تکنولوژی، آموزش‌های برنامه‌نویسی و نکات طراحی تجربه کاربری را اینجا بخوانید.
          </p>
        </div>

        {/* Search Bar */}
        <div className="max-w-2xl mx-auto mb-16 relative">
            <Search className="absolute right-4 top-3.5 text-slate-400" size={20} />
            <input 
                type="text" 
                placeholder="جستجو در مقالات..." 
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl px-12 py-3 outline-none focus:border-indigo-500 focus:ring-1 focus:ring-indigo-500 transition-all text-slate-700 dark:text-slate-200"
            />
        </div>

        {loading ? (
             <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {Array.from({ length: 6 }).map((_, i) => (
                  <BlogSkeleton key={i} />
                ))}
             </div>
        ) : (
            <>
                {posts.length === 0 ? (
                    <div className="text-center py-20 text-slate-500 dark:text-slate-400">
                        مقاله‌ای یافت نشد.
                    </div>
                ) : (
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                    {posts.map((post) => (
                        <a 
                        href={`#blog/${post.id}`}
                        onClick={(e) => handleNavClick(e, `#blog/${post.id}`)}
                        key={post.id} 
                        className="group bg-white dark:bg-slate-900 rounded-2xl overflow-hidden border border-slate-200 dark:border-slate-800 hover:shadow-2xl transition-all duration-300 flex flex-col animate-fade-in-up"
                        >
                        <div className="relative aspect-video overflow-hidden">
                            <img 
                            src={post.image} 
                            alt={post.title} 
                            className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-105"
                            />
                            <div className="absolute inset-0 bg-slate-900/10 group-hover:bg-slate-900/0 transition-colors"></div>
                        </div>
                        
                        <div className="p-6 flex-1 flex flex-col">
                            <div className="flex items-center gap-4 text-xs text-slate-500 dark:text-slate-400 mb-4">
                            <span className="flex items-center gap-1 bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded"><Calendar size={12} /> {post.date}</span>
                            <span className="flex items-center gap-1 bg-slate-100 dark:bg-slate-800 px-2 py-1 rounded"><Clock size={12} /> {post.readTime}</span>
                            </div>
                            
                            <h2 className="text-xl font-bold text-slate-900 dark:text-white mb-3 group-hover:text-indigo-600 dark:group-hover:text-indigo-400 transition-colors">
                            {post.title}
                            </h2>
                            
                            <p className="text-slate-600 dark:text-slate-400 text-sm leading-relaxed mb-6 flex-1">
                            {post.excerpt}
                            </p>
                            
                            <div className="flex items-center justify-between border-t border-slate-100 dark:border-slate-800 pt-4 mt-auto">
                            <span className="text-xs text-slate-500 dark:text-slate-500">نویسنده: {post.author}</span>
                            <span className="text-indigo-600 dark:text-indigo-400 text-sm font-medium flex items-center gap-1 group-hover:gap-2 transition-all">
                                مطالعه کامل
                                <ArrowLeft size={16} />
                            </span>
                            </div>
                        </div>
                        </a>
                    ))}
                    </div>
                )}

                {hasMore && (
                    <div className="flex justify-center mt-12">
                        <button 
                        onClick={loadMore} 
                        disabled={loadingMore}
                        className="px-8 py-3 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl text-slate-700 dark:text-slate-300 font-bold hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors flex items-center gap-2 shadow-sm disabled:opacity-50"
                        >
                        {loadingMore ? <Loader2 className="animate-spin" /> : <PlusCircle size={20} />}
                        مطالب قدیمی‌تر
                        </button>
                    </div>
                )}
            </>
        )}
      </div>
    </section>
  );
};

export default BlogList;