import React, { useEffect, useState, useRef } from 'react';

const CustomCursor: React.FC = () => {
  const [isVisible, setIsVisible] = useState(false);
  const [isHovering, setIsHovering] = useState(false);
  const [isClicking, setIsClicking] = useState(false);
  
  const cursorRef = useRef<HTMLDivElement>(null);
  const followerRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    // Determine if we should show the custom cursor
    // We target desktops (screen width > 1024px) to avoid issues on mobile/touch
    const checkVisibility = () => {
        const isLargeScreen = window.innerWidth >= 1024;
        const isTouch = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
        
        // Show if large screen and not primarily a touch device (though some laptops are both)
        if (isLargeScreen) {
            setIsVisible(true);
        } else {
            setIsVisible(false);
        }
    };

    checkVisibility();
    window.addEventListener('resize', checkVisibility);

    const onMouseMove = (e: MouseEvent) => {
      // Direct DOM manipulation for performance (avoids React render cycle lag)
      if (cursorRef.current) {
        cursorRef.current.style.transform = `translate3d(${e.clientX}px, ${e.clientY}px, 0)`;
      }
      if (followerRef.current) {
         followerRef.current.style.transform = `translate3d(${e.clientX}px, ${e.clientY}px, 0)`;
      }
    };

    const onMouseDown = () => setIsClicking(true);
    const onMouseUp = () => setIsClicking(false);

    const onMouseOver = (e: MouseEvent) => {
      const target = e.target as HTMLElement;
      
      // extensive check for interactive elements
      const isInteractive = 
        target.tagName === 'A' || 
        target.tagName === 'BUTTON' || 
        target.tagName === 'INPUT' || 
        target.tagName === 'TEXTAREA' || 
        target.tagName === 'SELECT' ||
        target.closest('a') || 
        target.closest('button') ||
        target.closest('[role="button"]') ||
        target.classList.contains('cursor-pointer') ||
        window.getComputedStyle(target).cursor === 'pointer';

      setIsHovering(!!isInteractive);
    };

    if (isVisible) {
        window.addEventListener('mousemove', onMouseMove);
        window.addEventListener('mousedown', onMouseDown);
        window.addEventListener('mouseup', onMouseUp);
        window.addEventListener('mouseover', onMouseOver);
    }

    return () => {
      window.removeEventListener('resize', checkVisibility);
      window.removeEventListener('mousemove', onMouseMove);
      window.removeEventListener('mousedown', onMouseDown);
      window.removeEventListener('mouseup', onMouseUp);
      window.removeEventListener('mouseover', onMouseOver);
    };
  }, [isVisible]);

  if (!isVisible) return null;

  return (
    <>
      <style>{`
        @media (min-width: 1024px) {
            * {
                cursor: none !important;
            }
        }
      `}</style>
      
      {/* Main Cursor Dot */}
      <div 
        ref={cursorRef}
        className="fixed top-0 left-0 w-3 h-3 bg-indigo-600 dark:bg-indigo-400 rounded-full pointer-events-none z-[10000] -ml-1.5 -mt-1.5 transition-transform duration-0"
        style={{ willChange: 'transform' }}
      />
      
      {/* Trailing Follower Circle */}
      <div 
        ref={followerRef}
        className={`fixed top-0 left-0 w-10 h-10 border border-indigo-500/50 dark:border-indigo-400/50 rounded-full pointer-events-none z-[9999] -ml-5 -mt-5 transition-all duration-200 ease-out ${
          isHovering ? 'scale-[2.5] bg-indigo-500/10 border-transparent backdrop-blur-[1px]' : 'scale-100'
        } ${isClicking ? '!scale-75 bg-indigo-600/20 !border-indigo-600' : ''}`}
        style={{ willChange: 'transform' }}
      />
    </>
  );
};

export default CustomCursor;