import React, { useEffect, useState } from 'react';
import { Phone, Sun, Moon } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AppSettings } from '../types';

interface MobileHeaderProps {
  isDark: boolean;
  toggleTheme: () => void;
}

const MobileHeader: React.FC<MobileHeaderProps> = ({ isDark, toggleTheme }) => {
  const [settings, setSettings] = useState<AppSettings | null>(null);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getSettings();
      setSettings(data);
    };
    update();
    window.addEventListener('settings-updated', update);
    return () => window.removeEventListener('settings-updated', update);
  }, []);

  const handleScrollTop = (e: React.MouseEvent) => {
    e.preventDefault();
    window.location.hash = '#home';
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  return (
    <div className="md:hidden fixed top-0 left-0 right-0 h-16 bg-white/90 dark:bg-slate-900/90 backdrop-blur-xl border-b border-slate-200 dark:border-slate-800 z-50 flex items-center justify-between px-4 shadow-sm dark:shadow-lg transition-all duration-300">
      
      {/* Controls (Right Side in RTL) */}
      <div className="flex items-center gap-3">
        <button 
          onClick={toggleTheme}
          className="w-9 h-9 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-400 hover:text-indigo-600 active:scale-95 transition-all"
        >
          {isDark ? <Sun size={18} /> : <Moon size={18} />}
        </button>

        <a 
          href="tel:02188881234" 
          className="w-9 h-9 rounded-full bg-slate-100 dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-600 dark:text-slate-400 hover:text-white hover:bg-indigo-600 dark:hover:bg-indigo-600 transition-all active:scale-95"
        >
          <Phone size={18} />
        </a>
      </div>

      {/* Logo (Left Side in RTL) */}
      <a href="#home" onClick={handleScrollTop} className="flex items-center gap-3">
        <div className="w-9 h-9 rounded-xl flex items-center justify-center overflow-hidden border border-white/10">
          {settings?.logoUrl ? (
             <img src={settings.logoUrl} alt="Logo" className="w-full h-full object-cover" />
           ) : (
             <div className="w-full h-full bg-gradient-to-br from-indigo-600 to-slate-800 flex items-center justify-center shadow-lg shadow-indigo-500/20">
               <span className="text-white font-black text-lg">V</span>
             </div>
           )}
        </div>
        <span className="text-lg font-bold text-slate-900 dark:text-white tracking-wide">{settings?.siteName || 'ویراوینگ'}</span>
      </a>
      
    </div>
  );
};

export default MobileHeader;