import React, { useState, useEffect } from 'react';
import { Sun, Moon } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AppSettings, MenuItem } from '../types';

interface NavbarProps {
  isDark: boolean;
  toggleTheme: () => void;
}

const Navbar: React.FC<NavbarProps> = ({ isDark, toggleTheme }) => {
  const [settings, setSettings] = useState<AppSettings | null>(null);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getSettings();
      setSettings(data);
    };
    update();
    window.addEventListener('settings-updated', update);
    return () => window.removeEventListener('settings-updated', update);
  }, []);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  const navLinks: MenuItem[] = settings?.navigation?.header || [];

  return (
    <nav className="fixed z-50 top-6 left-1/2 -translate-x-1/2 w-[95%] md:w-[90%] lg:w-[80%] max-w-7xl bg-white/90 dark:bg-slate-900/90 backdrop-blur-xl border border-slate-200 dark:border-slate-800 rounded-2xl shadow-2xl transition-all duration-300 hidden md:block">
      <div className="w-full flex items-center justify-between px-6 py-3">
        
        {/* Desktop Menu (Right Side in RTL) */}
        <div className="flex items-center gap-6">
          <ul className="flex flex-row space-x-8 rtl:space-x-reverse items-center font-medium">
            {navLinks.map((link) => {
              const target = link.anchor ? `${link.href}?scroll=${link.anchor}` : link.href;
              return (
                <li key={link.id}>
                  <a
                    href={target}
                    onClick={(e) => handleNavClick(e, target)}
                    className={`block transition-colors duration-200 cursor-pointer ${
                      link.special
                        ? 'bg-indigo-600 text-white px-5 py-2.5 rounded-xl hover:bg-indigo-700 shadow-lg shadow-indigo-500/20'
                        : 'text-slate-600 dark:text-gray-300 hover:text-indigo-600 dark:hover:text-white py-2'
                    }`}
                  >
                    {link.name}
                  </a>
                </li>
              );
            })}
          </ul>
          
          {/* Theme Toggle */}
          <button 
            onClick={toggleTheme}
            className="w-10 h-10 rounded-xl bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-400 hover:bg-indigo-100 dark:hover:bg-slate-700 hover:text-indigo-600 transition-colors"
          >
             {isDark ? <Sun size={20} /> : <Moon size={20} />}
          </button>
        </div>

        {/* Logo (Left Side in RTL) */}
        <a 
          href="#home" 
          onClick={(e) => handleNavClick(e, '#home')}
          className="flex items-center space-x-3 space-x-reverse rtl:space-x-reverse"
        >
          <div className="w-10 h-10 rounded-lg flex items-center justify-center overflow-hidden">
             {settings?.logoUrl ? (
               <img src={settings.logoUrl} alt="Logo" className="w-full h-full object-cover" />
             ) : (
               <div className="w-full h-full bg-gradient-to-br from-indigo-600 to-slate-800 flex items-center justify-center shadow-lg shadow-indigo-500/20">
                 <span className="text-white font-bold text-xl">V</span>
               </div>
             )}
          </div>
          <span className="self-center text-2xl font-bold whitespace-nowrap text-slate-900 dark:text-white tracking-wide">
            {settings?.siteName || '...'}
          </span>
        </a>

      </div>
    </nav>
  );
};

export default Navbar;