import React, { useState, useEffect } from 'react';
import { Globe, Smartphone, Palette, Search, Server, ShieldCheck, Zap, Code, Database, Layout } from 'lucide-react';
import { DataService } from '../services/dataService';
import { Service } from '../types';

const iconMap: Record<string, React.ReactNode> = {
  'Globe': <Globe className="w-8 h-8 text-blue-500 dark:text-blue-400" />,
  'Smartphone': <Smartphone className="w-8 h-8 text-green-500 dark:text-green-400" />,
  'Palette': <Palette className="w-8 h-8 text-pink-500 dark:text-pink-400" />,
  'Search': <Search className="w-8 h-8 text-yellow-500 dark:text-yellow-400" />,
  'Server': <Server className="w-8 h-8 text-indigo-500 dark:text-indigo-400" />,
  'ShieldCheck': <ShieldCheck className="w-8 h-8 text-red-500 dark:text-red-400" />,
  'Zap': <Zap className="w-8 h-8 text-orange-500 dark:text-orange-400" />,
  'Code': <Code className="w-8 h-8 text-teal-500 dark:text-teal-400" />,
  'Database': <Database className="w-8 h-8 text-cyan-500 dark:text-cyan-400" />,
  'Layout': <Layout className="w-8 h-8 text-purple-500 dark:text-purple-400" />,
};

const Services: React.FC = () => {
  const [services, setServices] = useState<Service[]>([]);

  useEffect(() => {
    const updateServices = async () => {
      const data = await DataService.getServices();
      setServices(data);
    };
    updateServices();
    window.addEventListener('data-updated', updateServices);
    return () => window.removeEventListener('data-updated', updateServices);
  }, []);

  return (
    <section id="services" className="py-16 md:py-24 bg-slate-50 dark:bg-slate-900 relative transition-colors duration-300">
      <div className="container mx-auto md:px-4">
        <div className="text-center mb-8 md:mb-16 px-4">
          <h2 className="text-2xl md:text-5xl font-bold text-slate-900 dark:text-white mb-4">خدمات ما</h2>
          <div className="h-1 w-24 bg-indigo-600 mx-auto rounded-full hidden md:block mb-6"></div>
          <p className="text-slate-600 dark:text-slate-400 max-w-2xl mx-auto text-sm md:text-lg">
            راهکارهای جامع برای رشد کسب‌وکـار شما
          </p>
        </div>

        {/* Mobile: Horizontal Scroll | Desktop: Grid */}
        <div className="flex overflow-x-auto gap-4 px-4 pb-12 snap-x snap-mandatory hide-scrollbar md:grid md:grid-cols-2 lg:grid-cols-3 md:gap-8 md:pb-4 md:pt-2">
          {services.map((service) => (
            <div 
              key={service.id}
              className="min-w-[260px] w-[75vw] md:w-auto snap-center flex-shrink-0 group p-6 md:p-8 bg-white dark:bg-slate-800/40 border border-slate-200 dark:border-slate-700 rounded-3xl hover:bg-white dark:hover:bg-slate-800 transition-all duration-300 md:hover:-translate-y-1 hover:shadow-xl hover:shadow-indigo-500/10 dark:hover:shadow-indigo-500/10 flex flex-col items-start active:scale-[0.98] md:active:scale-100"
            >
              <div className="mb-4 md:mb-6 p-4 bg-slate-50 dark:bg-slate-900/80 rounded-2xl inline-block group-hover:scale-110 transition-transform duration-300 border border-slate-100 dark:border-slate-700/50">
                {iconMap[service.iconName] || iconMap['Globe']}
              </div>
              <h3 className="text-lg md:text-xl font-bold text-slate-900 dark:text-white mb-2 group-hover:text-indigo-600 dark:group-hover:text-indigo-400 transition-colors">
                {service.title}
              </h3>
              <p className="text-slate-600 dark:text-slate-400 leading-relaxed text-sm md:text-base">
                {service.description}
              </p>
            </div>
          ))}
          {/* Spacer for mobile scroll */}
          <div className="w-1 flex-shrink-0 md:hidden"></div>
        </div>
      </div>
    </section>
  );
};

export default Services;