import React, { useState, useEffect, Suspense } from 'react';
import Navbar from '../components/Navbar';
import MobileHeader from '../components/MobileHeader';
import BottomNav from '../components/BottomNav';
import Footer from '../components/Footer';
import ToastContainer from '../components/Toast';
import Installer from '../components/Installer';
import HomeView from '../components/HomeView';
import ErrorBoundary from '../components/ErrorBoundary';
import { DataService } from '../services/dataService';
import { Loader2 } from 'lucide-react';
import NotFound from '../components/NotFound';
import ScrollToTop from '../components/ScrollToTop';
import CustomCursor from '../components/CustomCursor';

// Lazy load other pages to reduce initial bundle size
const Portfolio = React.lazy(() => import('../components/Portfolio'));
const ProjectDetail = React.lazy(() => import('../components/ProjectDetail'));
const Contact = React.lazy(() => import('../components/Contact'));
const AdminPanel = React.lazy(() => import('../components/AdminPanel'));
const BlogList = React.lazy(() => import('../components/BlogList'));
const BlogPostDetail = React.lazy(() => import('../components/BlogPost'));
const Login = React.lazy(() => import('../components/Login'));
const LegalPage = React.lazy(() => import('../components/LegalPage'));

const PageLoader: React.FC = () => (
  <div className="min-h-[60vh] flex flex-col items-center justify-center">
    <Loader2 className="w-10 h-10 text-indigo-600 animate-spin mb-4" />
    <p className="text-slate-500 text-sm animate-pulse">در حال بارگذاری...</p>
  </div>
);

const App: React.FC = () => {
  const [route, setRoute] = useState(window.location.hash || '#home');
  const [isDark, setIsDark] = useState(true);
  const [isInstalled, setIsInstalled] = useState(false);
  const [isLoading, setIsLoading] = useState(true);

  // Initial Checks
  useEffect(() => {
    const checkInstall = async () => {
      const installed = await DataService.isInstalled();
      setIsInstalled(installed);
      setIsLoading(false);
    };

    checkInstall();
    window.addEventListener('app-installed', () => {
      checkInstall();
    });

    // Check Theme
    const savedTheme = localStorage.getItem('theme');
    const systemPrefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    if (savedTheme === 'dark' || (!savedTheme && systemPrefersDark)) {
      setIsDark(true);
      document.documentElement.classList.add('dark');
    } else {
      setIsDark(false);
      document.documentElement.classList.remove('dark');
    }

    return () => window.removeEventListener('app-installed', checkInstall);
  }, []);

  const toggleTheme = () => {
    const newMode = !isDark;
    setIsDark(newMode);
    if (newMode) {
      document.documentElement.classList.add('dark');
      localStorage.setItem('theme', 'dark');
    } else {
      document.documentElement.classList.remove('dark');
      localStorage.setItem('theme', 'light');
    }
  };

  useEffect(() => {
    const handleHashChange = () => {
      const newHash = window.location.hash || '#home';
      setRoute(newHash);
      if (!newHash.includes('?scroll=')) {
        window.scrollTo(0, 0);
      }
    };
    window.addEventListener('hashchange', handleHashChange);
    return () => window.removeEventListener('hashchange', handleHashChange);
  }, []);

  // Comprehensive SEO Logic
  useEffect(() => {
    const updateSEO = async () => {
      try {
        const settings = await DataService.getSettings();
        if (!settings?.seo) return;

        const seo = settings.seo;

        // Helper function to set or update meta tag
        const setMetaTag = (name: string, content: string, attribute: string = 'name') => {
          if (!content) return;
          let meta = document.querySelector(`meta[${attribute}="${name}"]`);
          if (!meta) {
            meta = document.createElement('meta');
            meta.setAttribute(attribute, name);
            document.head.appendChild(meta);
          }
          meta.setAttribute('content', content);
        };

        // Basic Meta Tags
        if (seo.title) document.title = seo.title;
        setMetaTag('description', seo.description);
        setMetaTag('keywords', Array.isArray(seo.keywords) ? seo.keywords.join(', ') : '');
        setMetaTag('author', seo.author || '');
        setMetaTag('robots', seo.robots || 'index, follow');
        if (seo.canonicalUrl) {
          let canonical = document.querySelector('link[rel="canonical"]');
          if (!canonical) {
            canonical = document.createElement('link');
            canonical.setAttribute('rel', 'canonical');
            document.head.appendChild(canonical);
          }
          canonical.setAttribute('href', seo.canonicalUrl);
        }

        // Open Graph Tags
        setMetaTag('og:title', seo.ogTitle || seo.title, 'property');
        setMetaTag('og:description', seo.ogDescription || seo.description, 'property');
        setMetaTag('og:image', seo.ogImage || '', 'property');
        setMetaTag('og:url', seo.ogUrl || seo.canonicalUrl || window.location.href, 'property');
        setMetaTag('og:type', seo.ogType || 'website', 'property');
        setMetaTag('og:site_name', seo.ogSiteName || settings.siteName, 'property');
        setMetaTag('og:locale', seo.ogLocale || 'fa_IR', 'property');

        // Twitter Card Tags
        setMetaTag('twitter:card', seo.twitterCard || 'summary_large_image');
        setMetaTag('twitter:site', seo.twitterSite || '');
        setMetaTag('twitter:creator', seo.twitterCreator || '');
        setMetaTag('twitter:title', seo.twitterTitle || seo.ogTitle || seo.title);
        setMetaTag('twitter:description', seo.twitterDescription || seo.ogDescription || seo.description);
        setMetaTag('twitter:image', seo.twitterImage || seo.ogImage || '');

        // Theme Color
        if (seo.themeColor) {
          setMetaTag('theme-color', seo.themeColor);
        }

        // Google Analytics
        if (seo.googleAnalyticsId) {
          // Remove existing GA script
          const existingGA = document.querySelector('script[data-ga]');
          if (existingGA) existingGA.remove();

          // Add new GA script
          const gaScript = document.createElement('script');
          gaScript.setAttribute('async', '');
          gaScript.setAttribute('src', `https://www.googletagmanager.com/gtag/js?id=${seo.googleAnalyticsId}`);
          gaScript.setAttribute('data-ga', 'true');
          document.head.appendChild(gaScript);

          const gaInit = document.createElement('script');
          gaInit.setAttribute('data-ga', 'true');
          gaInit.textContent = `
            window.dataLayer = window.dataLayer || [];
            function gtag(){dataLayer.push(arguments);}
            gtag('js', new Date());
            gtag('config', '${seo.googleAnalyticsId}');
          `;
          document.head.appendChild(gaInit);
        }

        // Verification Codes
        if (seo.googleSearchConsole) {
          setMetaTag('google-site-verification', seo.googleSearchConsole);
        }
        if (seo.bingVerification) {
          setMetaTag('msvalidate.01', seo.bingVerification);
        }
        if (seo.yandexVerification) {
          setMetaTag('yandex-verification', seo.yandexVerification);
        }

        // Schema.org JSON-LD
        if (seo.schemaType && seo.schemaName) {
          // Remove existing schema
          const existingSchema = document.querySelector('script[type="application/ld+json"]');
          if (existingSchema) existingSchema.remove();

          const schema: any = {
            '@context': 'https://schema.org',
            '@type': seo.schemaType,
            name: seo.schemaName,
            description: seo.schemaDescription || seo.description,
            url: seo.schemaUrl || seo.canonicalUrl || window.location.origin,
          };

          if (seo.schemaLogo) schema.logo = seo.schemaLogo;
          if (seo.schemaEmail) schema.email = seo.schemaEmail;
          if (seo.schemaPhone) schema.telephone = seo.schemaPhone;
          if (seo.schemaSameAs && seo.schemaSameAs.length > 0) {
            schema.sameAs = seo.schemaSameAs;
          }
          if (seo.schemaAddress) {
            schema.address = {
              '@type': 'PostalAddress',
              ...seo.schemaAddress
            };
          }

          const schemaScript = document.createElement('script');
          schemaScript.setAttribute('type', 'application/ld+json');
          schemaScript.textContent = JSON.stringify(schema);
          document.head.appendChild(schemaScript);
        }
      } catch (e) { 
        console.error('SEO update error:', e); 
      }
    };
    updateSEO();
    window.addEventListener('settings-updated', updateSEO);
    return () => window.removeEventListener('settings-updated', updateSEO);
  }, []);

  if (isLoading) return null;

  if (!isInstalled) {
    return <Installer />;
  }

  // Simple Router Logic
  let Component;
  let showFooter = true;
  let showNavbar = true;

  const isHome = route === '#home' || route.startsWith('#home?');
  const isAdmin = route.startsWith('#admin');
  const isProject = route.startsWith('#project/');
  const isBlogPost = route.startsWith('#blog/');
  const isBlogList = route === '#blog' || route.startsWith('#blog?');
  const isPortfolio = route === '#portfolio';
  const isContact = route.startsWith('#contact');
  const isTerms = route.startsWith('#terms');
  const isPrivacy = route.startsWith('#privacy');

  if (isAdmin) {
    showFooter = false;
    showNavbar = false;
    Component = DataService.isAuthenticated() ? <AdminPanel /> : <Login />;
  } else if (isProject) {
    const projectId = route.split('/')[1];
    Component = <ProjectDetail id={projectId} />;
  } else if (isBlogPost) {
    const blogId = route.split('/')[1];
    Component = <BlogPostDetail id={blogId} />;
  } else if (isBlogList) {
    Component = <BlogList />;
  } else if (isPortfolio) {
    Component = <Portfolio />;
  } else if (isContact) {
    Component = <Contact />;
  } else if (isTerms) {
    Component = <LegalPage type="terms" />;
  } else if (isPrivacy) {
    Component = <LegalPage type="privacy" />;
  } else if (isHome) {
    Component = <HomeView />;
  } else {
    Component = <NotFound />;
    showFooter = false;
  }

  return (
    <div className="min-h-screen bg-slate-50 dark:bg-slate-900 text-slate-900 dark:text-slate-50 selection:bg-indigo-50 selection:text-white pb-16 md:pb-0 transition-colors duration-300">
      <ErrorBoundary>
        <ToastContainer />
        <ScrollToTop />
        <CustomCursor />
        {showNavbar && <Navbar isDark={isDark} toggleTheme={toggleTheme} />}
        {showNavbar && <MobileHeader isDark={isDark} toggleTheme={toggleTheme} />}
        
        <main>
          <Suspense fallback={<PageLoader />}>
            {Component}
          </Suspense>
        </main>

        {showFooter && !isContact && !isProject && (
          <Footer />
        )}
        
        {showNavbar && <BottomNav currentRoute={route} />}
      </ErrorBoundary>
    </div>
  );
};

export default App;

